/*
 *   This file is part of Skript.
 *
 *  Skript is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Skript is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Skript.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * 
 * Copyright 2011, 2012 Peter Gttinger
 * 
 */

package ch.njol.skript.util;

import java.lang.reflect.Array;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;

import org.bukkit.block.BlockFace;
import org.bukkit.entity.Animals;
import org.bukkit.entity.Arrow;
import org.bukkit.entity.Boat;
import org.bukkit.entity.CaveSpider;
import org.bukkit.entity.Chicken;
import org.bukkit.entity.Cow;
import org.bukkit.entity.Creature;
import org.bukkit.entity.Creeper;
import org.bukkit.entity.Egg;
import org.bukkit.entity.Enderman;
import org.bukkit.entity.Entity;
import org.bukkit.entity.ExperienceOrb;
import org.bukkit.entity.Explosive;
import org.bukkit.entity.FallingSand;
import org.bukkit.entity.Fireball;
import org.bukkit.entity.Fish;
import org.bukkit.entity.Flying;
import org.bukkit.entity.Ghast;
import org.bukkit.entity.Giant;
import org.bukkit.entity.HumanEntity;
import org.bukkit.entity.Item;
import org.bukkit.entity.LightningStrike;
import org.bukkit.entity.LivingEntity;
import org.bukkit.entity.Minecart;
import org.bukkit.entity.Monster;
import org.bukkit.entity.Painting;
import org.bukkit.entity.Pig;
import org.bukkit.entity.PigZombie;
import org.bukkit.entity.Player;
import org.bukkit.entity.PoweredMinecart;
import org.bukkit.entity.Projectile;
import org.bukkit.entity.Sheep;
import org.bukkit.entity.Silverfish;
import org.bukkit.entity.Skeleton;
import org.bukkit.entity.Slime;
import org.bukkit.entity.Snowball;
import org.bukkit.entity.Spider;
import org.bukkit.entity.Squid;
import org.bukkit.entity.StorageMinecart;
import org.bukkit.entity.TNTPrimed;
import org.bukkit.entity.Vehicle;
import org.bukkit.entity.WaterMob;
import org.bukkit.entity.Wolf;
import org.bukkit.entity.Zombie;
import org.bukkit.event.Event;
import org.bukkit.event.entity.EntityDamageByBlockEvent;
import org.bukkit.event.entity.EntityDamageByEntityEvent;
import org.bukkit.event.vehicle.VehicleDamageEvent;
import org.bukkit.event.vehicle.VehicleDestroyEvent;
import org.bukkit.inventory.Inventory;
import org.bukkit.inventory.ItemStack;
import org.bukkit.util.Vector;

import ch.njol.skript.Skript;
import ch.njol.skript.SkriptLogger;

/**
 * 
 * Utility class. these functions are really useful if you intend to make new conditions, effects and variables.
 * 
 * @author Peter Gttinger
 * 
 */
public abstract class Utils {
	private Utils() {}
	
	public static boolean parseBoolean(final String s) {
		boolean r = (s.equalsIgnoreCase("true") || s.equalsIgnoreCase("yes"));
		if (r)
			return true;
		r = s.equalsIgnoreCase("false") || s.equalsIgnoreCase("no");
		if (r)
			return false;
		SkriptLogger.expectationError("boolean (true/yes or false/no)", s);
		return false;
	}
	
	public static int parseBooleanNoError(final String s) {
		if (s.equalsIgnoreCase("true") || s.equalsIgnoreCase("yes"))
			return 1;
		if (s.equalsIgnoreCase("false") || s.equalsIgnoreCase("no"))
			return 0;
		return -1;
	}
	
	public static short parseShort(final String s) {
		try {
			return Short.parseShort(s);
		} catch (final NumberFormatException e) {
			SkriptLogger.expectationError("integer between -32768 and 32767", s);
		}
		return 0;
	}
	
	public static int parseInt(final String s) {
		try {
			return Integer.parseInt(s);
		} catch (final NumberFormatException e) {
			SkriptLogger.expectationError("integer", s);
		}
		return 0;
	}
	
	public static float parseFloat(final String s) {
		try {
			return Float.parseFloat(s);
		} catch (final NumberFormatException e) {
			SkriptLogger.expectationError("number", s);
		}
		return 0;
	}
	
	public static double parseDouble(final String s) {
		try {
			return Double.parseDouble(s);
		} catch (final NumberFormatException e) {
			SkriptLogger.expectationError("number", s);
		}
		return 0;
	}
	
	public static <T> int contains(final T[] array, final T o) {
		if (array == null)
			return -1;
		int i = 0;
		for (final T a : array) {
			if (a.equals(o))
				return i;
			i++;
		}
		return -1;
	}
	
	public static int containsIgnoreCase(final String[] array, final String s) {
		if (array == null)
			return -1;
		int i = 0;
		for (final String a : array) {
			if (a.equalsIgnoreCase(s))
				return i;
			i++;
		}
		return -1;
	}
	
	/**
	 * 
	 * Finds an object in an array using {@link Object#equals(Object)} to test equality.
	 * 
	 * @param array
	 * @param o
	 * @return
	 */
	public static <T> int contains(final Iterable<T> array, final T o) {
		if (array == null)
			return -1;
		int i = 0;
		for (final T a : array) {
			if (a.equals(o))
				return i;
			i++;
		}
		return -1;
	}
	
	/**
	 * finds a String in an array of Strings (ignoring case).
	 * 
	 * @param array
	 * @param s
	 * @return
	 */
	public static int containsIgnoreCase(final Iterable<String> array, final String s) {
		if (array == null)
			return -1;
		int i = 0;
		for (final String a : array) {
			if (a.equalsIgnoreCase(s))
				return i;
			i++;
		}
		return -1;
	}
	
	public static <T, U> Entry<T, U> containsKey(final Map<T, U> m, final Object key) {
		if (m == null)
			return null;
		for (final Entry<T, U> e : m.entrySet()) {
			if (e.getKey() == key)
				return e;
		}
		return null;
	}
	
	public static <T, U> Entry<T, U> containsEqualKey(final Map<T, U> m, final Object key) {
		if (m == null)
			return null;
		for (final Entry<T, U> e : m.entrySet()) {
			if (e.getKey().equals(key))
				return e;
		}
		return null;
	}
	
	public static <U> Entry<String, U> containsKeyIgnoreCase(final Map<String, U> m, final String key) {
		if (m == null)
			return null;
		for (final Entry<String, U> e : m.entrySet()) {
			if (e.getKey().equalsIgnoreCase(key))
				return e;
		}
		return null;
	}
	
	public static BlockFace getBlockFace(String s, final boolean printError) {
		if (s == null || s.isEmpty())
			return null;
		s = s.toUpperCase(Locale.ENGLISH).replace(' ', '_');
		try {
			if (s.equals("ABOVE"))
				return BlockFace.UP;
			if (s.equals("BELOW"))
				return BlockFace.DOWN;
			return BlockFace.valueOf(s);
		} catch (final IllegalArgumentException e1) {
			if (s.equals("U"))
				return BlockFace.UP;
			if (s.equals("D"))
				return BlockFace.DOWN;
			if (s.length() <= 3) {
				try {
					String r = "";
					for (int i = 0; i < s.length(); i++) {
						switch (s.charAt(i)) {
							case 'n':
								r += "NORTH_";
							break;
							case 'e':
								r += "EAST_";
							break;
							case 's':
								r += "SOUTH_";
							break;
							case 'w':
								r += "WEST_";
							break;
							default:
								if (printError)
									Skript.error("invalid direction '" + s + "'");
								return null;
						}
					}
					return BlockFace.valueOf(r.substring(0, r.length() - 1));
				} catch (final IllegalArgumentException e2) {}
			}
		}
		if (printError)
			Skript.error("invalid direction '" + s + "'");
		return null;
	}
	
	public static final int[] getBlockFaceDir(final BlockFace f) {
		return new int[] {f.getModX(), f.getModY(), f.getModZ()};
	}
	
	public static final int getBlockFaceDir(final BlockFace f, final int axis) {
		return getBlockFaceDir(f)[axis];
	}
	
	public static EntityType getEntityType(String s) {
		s = s.toLowerCase(Locale.ENGLISH);
		int amount = 1;
		if (s.matches("\\d+ .+")) {
			amount = Integer.parseInt(s.split(" ", 2)[0]);
			s = s.split(" ", 2)[1];
		} else if (s.matches("an? .*")) {
			s = s.split(" ", 2)[1];
		}
		if (amount > 1 && s.endsWith("s"))
			s = s.substring(0, s.length() - 1);
		final Class<? extends Entity> c = getEntityClass(s, amount == 1);
		if (c == null)
			return null;
		return new EntityType(c, amount);
	}
	
	private static Class<? extends Entity> getEntityClass(final String name, final boolean singular) {
		if (name.equals("any") || name.equals("entity") || !singular && name.equals("entities")) {
			return Entity.class;
		} else if (name.equals("animal")) {
			return Animals.class;
		} else if (name.equals("arrow")) {
			return Arrow.class;
		} else if (name.equals("boat")) {
			return Boat.class;
		} else if (name.equals("chicken")) {
			return Chicken.class;
		} else if (name.equals("cow")) {
			return Cow.class;
		} else if (name.equals("creature")) {
			return Creature.class;
		} else if (name.equals("creeper")) {
			return Creeper.class;
		} else if (name.equals("egg")) {
			return Egg.class;
		} else if (name.equals("explosive")) {
			return Explosive.class;
		} else if (name.equals("falling")) {
			return FallingSand.class;
		} else if (name.equals("fireball")) {
			return Fireball.class;
		} else if (name.equals("fish") || !singular && name.equals("fishe")) {// fishes
			return Fish.class;
		} else if (name.equals("flying")) {
			return Flying.class;
		} else if (name.equals("ghast")) {
			return Ghast.class;
		} else if (name.equals("giant")) {
			return Giant.class;
		} else if (name.equals("human")) {
			return HumanEntity.class;
		} else if (name.equals("item")) {
			return Item.class;
		} else if (name.equals("lightning strike")) {
			return LightningStrike.class;
		} else if (name.equals("living")) {
			return LivingEntity.class;
		} else if (name.equals("minecart")) {
			return Minecart.class;
		} else if (name.equals("monster")) {
			return Monster.class;
		} else if (name.equals("painting")) {
			return Painting.class;
		} else if (name.equals("pig")) {
			return Pig.class;
		} else if (name.equals("pigzombie")) {
			return PigZombie.class;
		} else if (name.equals("player")) {
			return Player.class;
		} else if (name.equals("powered minecart")) {
			return PoweredMinecart.class;
		} else if (name.equals("projectile")) {
			return Projectile.class;
		} else if (name.equals("sheep")) {
			return Sheep.class;
		} else if (name.equals("skeleton")) {
			return Skeleton.class;
		} else if (name.equals("slime")) {
			return Slime.class;
		} else if (name.equals("snowball")) {
			return Snowball.class;
		} else if (name.equals("spider")) {
			return Spider.class;
		} else if (name.equals("squid")) {
			return Squid.class;
		} else if (name.equals("storage minecart")) {
			return StorageMinecart.class;
		} else if (name.equals("tnt")) {
			return TNTPrimed.class;
		} else if (name.equals("vehicle")) {
			return Vehicle.class;
		} else if (name.equals("water")) {
			return WaterMob.class;
		} else if (name.equals("wolf") || !singular && name.equals("wolve")) {// wolves
			return Wolf.class;
		} else if (name.equals("zombie")) {
			return Zombie.class;
		} else if (name.equals("cave spider")) {
			return CaveSpider.class;
		} else if (name.equals("enderman") || !singular && name.equals("endermen")) {
			return Enderman.class;
		} else if (name.equals("experience orb")) {
			return ExperienceOrb.class;
		} else if (name.equals("silverfish") || !singular && name.equals("silverfishe")) {// silverfishes
			return Silverfish.class;
		}
		return null;
	}
	
	/**
	 * Parses an ItemType to be used as an alias, i.e. it doesn't parse 'all'/'every' and the amount.
	 * 
	 * @param s
	 * @return
	 */
	public static ItemType parseAlias(String s) {
		if (s == null || s.isEmpty())
			return null;
		if (s.equals("*"))
			return Skript.everything;
		
		s = s.toLowerCase(Locale.ENGLISH);
		
		final ItemType t = new ItemType();
		
		final String[] types = s.split("\\s*,\\s*");
		for (final String type : types) {
			if (parseType(type, t) == null)
				return null;
		}
		
		return t;
	}
	
	/**
	 * Parses an ItemType
	 * 
	 * @param s
	 * @return The parsed ItemType or null if the input is invalid.
	 */
	public static ItemType parseItemType(String s) {
		if (s == null || s.isEmpty())
			return null;
		s = s.toLowerCase();
		if (s.contains(",") || s.contains(" and ") || s.contains(" or "))
			return null;
//			throw new SkriptAPIException("Invalid method call");
		
		final ItemType t = new ItemType();
		
		if (s.matches("\\d+ of (all|every) .+")) {
			t.amount = parseInt(s.split(" ", 2)[0]);
			t.all = true;
			s = s.split(" ", 4)[3];
		} else if (s.matches("\\d+ (of )?.+")) {
			t.amount = parseInt(s.split(" ", 2)[0]);
			if (s.matches("\\d+ of .+"))
				s = s.split(" ", 3)[2];
			else
				s = s.split(" ", 2)[1];
		} else if (s.matches("an? .+")) {
			t.amount = 1;
			s = s.split(" ", 2)[1];
		} else if (s.matches("(all|every) .+")) {
			t.all = true;
			s = s.split(" ", 2)[1];
		}
		
		parseType(s, t);
		
		if (!t.hasTypes())
			return null;
		
		return t;
	}
	
	/**
	 * 
	 * @param s The string holding the type, can be either a number or an alias, plus an optional data part.
	 * @param t The ItemType to add the parsed ItemData(s) to (i.e. this ItemType will be modified)
	 * @return The given item type or null if the input couldn't be parsed.
	 */
	private final static ItemType parseType(final String s, final ItemType t) {
		ItemType i;
		int c = s.indexOf(':');
		if (c == -1)
			c = s.length();
		final String typeR = s.substring(0, c);
		ItemData data = null;
		if (c != s.length()) {
			data = parseData(s.substring(c + 1));
			if (data == null) {
				Skript.setErrorCause("'" + s.substring(c) + "' is no a valid item data", false);
				return null;
			}
		}
		if (typeR.isEmpty()) {
			t.add(data);
			return t;
		} else if (typeR.matches("\\d+")) {
			ItemData d = new ItemData();
			d.typeid = parseInt(typeR);
			if (data != null)
				d = d.intersection(data);
			t.add(d);
			return t;
		} else if ((i = getAlias(typeR, t.amount == 1 && !t.all)) != null) {
			for (ItemData d : i) {
				if (data != null)
					d = d.intersection(data);
				t.add(d);
			}
			return t;
		}
		Skript.setErrorCause("'" + s + "' is neither an id nor an alias", false);
		return null;
	}
	
	/**
	 * Gets an alias from the aliases defined in the config.
	 * 
	 * @param s The alias to get
	 * @param singular if false this method will erase common plural endings
	 * @return The ItemType associated with the given alias or null if no such alias exists.
	 */
	private final static ItemType getAlias(final String s, final boolean singular) {
		ItemType i;
		if ((i = Skript.aliases.get(s)) != null)
			return i.clone();
		if (!singular) {
			if (s.endsWith("s")) {
				if ((i = Skript.aliases.get(s.substring(0, s.length() - 1))) != null)
					return i.clone();
				if (s.endsWith("es") && (i = Skript.aliases.get(s.substring(0, s.length() - 2))) != null)
					return i.clone();
				if (s.endsWith("ves") && (i = Skript.aliases.get(s.substring(0, s.length() - 3) + "f")) != null)
					return i.clone();
			} else if (s.endsWith("i") && (i = Skript.aliases.get(s.substring(0, s.length() - 1) + "us")) != null) {
				return i.clone();
			}
		}
		if (s.startsWith("any ")) {
			return getAlias(s.substring("any ".length()), singular);
		}
		return null;
	}
	
	/**
	 * gets the data part of an item data
	 * 
	 * @param s Everything after & not including ':'
	 * @return ItemData with only the dataMin and dataMax set
	 */
	private final static ItemData parseData(final String s) {
		if (!s.matches("(\\d+)?(-(\\d+)?)?"))
			return null;
		final ItemData t = new ItemData();
		int i = s.indexOf('-');
		if (i == -1)
			i = s.length();
		t.dataMin = (i == 0 ? -1 : parseShort(s.substring(0, i)));
		t.dataMax = (i == s.length() ? t.dataMin : (i == s.length() - 1 ? -1 : parseShort(s.substring(i + 1, s.length()))));
		return t;
	}
	
	public static boolean inventoryContains(final Inventory inventory, final ItemStack[] items) {
		for (final ItemStack item : items) {
			if (!inventoryContains(inventory, item))
				return false;
		}
		return true;
	}
	
	public static boolean inventoryContains(final Inventory inventory, final ItemStack item) {
		int amount = 0;
		for (final ItemStack i : inventory.all(item.getType()).values()) {
			if (!(i.getData() == null ^ item.getData() == null) && (i.getData() == null && item.getData() == null || i.getData().getData() == item.getData().getData()))
				amount += i.getAmount();
		}
		return amount >= item.getAmount();
	}
	
	public static boolean removeItemStacks(final Inventory inventory, final ItemStack[] items, final Player player) {
		return removeItemStacks(inventory, items, player);
	}
	
	@SuppressWarnings("deprecation")
	public static boolean removeItemStacks(final Inventory inventory, final Iterable<ItemStack> items, final Player player) {
		boolean ok = true;
		for (final ItemStack item : items) {
			ok &= removeItemStack(inventory, item, null);
		}
		if (player != null) {
			player.updateInventory();
		}
		return ok;
	}
	
	@SuppressWarnings("deprecation")
	public static boolean removeItemStack(final List<ItemStack> items, final ItemStack item, final Player player) {
		int amount = item.getAmount();
		for (int i = 0; i < items.size(); i++) {
			final ItemStack is = items.get(i);
			if (is == null || is.getTypeId() != item.getTypeId())
				continue;
			if (!(is.getData() == null ^ item.getData() == null) && (is.getData() == null && item.getData() == null || is.getData().getData() == item.getData().getData())) {
				if (is.getAmount() > amount) {
					is.setAmount(is.getAmount() - amount);
					if (player != null) {
						player.updateInventory();
					}
					return true;
				}
				amount -= is.getAmount();
				items.remove(i);
				if (amount == 0) {
					if (player != null) {
						player.updateInventory();
					}
					return true;
				}
			}
		}
		return false;
	}
	
	@SuppressWarnings("deprecation")
	public static boolean removeItemStack(final Inventory inventory, final ItemStack item, final Player player) {
		int amount = item.getAmount();
		for (int i = 0; i < inventory.getSize(); i++) {
			final ItemStack is = inventory.getItem(i);
			if (is == null || is.getTypeId() != item.getTypeId())
				continue;
			if (!(is.getData() == null ^ item.getData() == null) && (is.getData() == null && item.getData() == null || is.getData().getData() == item.getData().getData())) {
				if (is.getAmount() > amount) {
					is.setAmount(is.getAmount() - amount);
					if (player != null) {
						player.updateInventory();
					}
					return true;
				}
				amount -= is.getAmount();
				inventory.clear(i);
				if (amount == 0) {
					if (player != null) {
						player.updateInventory();
					}
					return true;
				}
			}
		}
		return false;
	}
	
	public static String join(final VariableString[] strings, final Event e) {
		final StringBuilder b = new StringBuilder();
		for (int i = 0; i < strings.length; i++) {
			if (i != 0)
				b.append(", ");
			b.append(strings[i].getDebugMessage(e));
		}
		return b.toString();
	}
	
	public static String join(final String[] strings) {
		return join(strings, ", ");
	}
	
	public static String join(final String[] strings, final String delimiter) {
		return join(strings, delimiter, 0, strings.length);
	}
	
	public static String join(final String[] strings, final String delimiter, final int start, final int end) {
		final StringBuilder b = new StringBuilder();
		for (int i = start; i < end; i++) {
			if (i != 0)
				b.append(delimiter);
			b.append(strings[i]);
		}
		return b.toString();
	}
	
	public static String join(final Object[] objects) {
		final StringBuilder b = new StringBuilder();
		for (int i = 0; i < objects.length; i++) {
			if (i != 0)
				b.append(", ");
			b.append(Skript.toString(objects[i]));
		}
		return b.toString();
	}
	
	public static String join(final List<?> objects) {
		final StringBuilder b = new StringBuilder();
		for (int i = 0; i < objects.size(); i++) {
			if (i != 0)
				b.append(", ");
			b.append(Skript.toString(objects.get(i)));
		}
		return b.toString();
	}
	
	public static Object getAttacker(final Event e) {
		if (e instanceof EntityDamageByEntityEvent) {
			if (((EntityDamageByEntityEvent) e).getDamager() instanceof Projectile) {
				return ((Projectile) ((EntityDamageByEntityEvent) e).getDamager()).getShooter();
			}
			return ((EntityDamageByEntityEvent) e).getDamager();
		} else if (e instanceof EntityDamageByBlockEvent) {
			return ((EntityDamageByBlockEvent) e).getDamager();
		} else if (e instanceof VehicleDamageEvent) {
			return ((VehicleDamageEvent) e).getAttacker();
		} else if (e instanceof VehicleDestroyEvent) {
			return ((VehicleDestroyEvent) e).getAttacker();
		}
		return null;
	}
	
	public static <T> T getRandom(final T[] os) {
		if (os == null || os.length == 0)
			return null;
		return os[Skript.random.nextInt(os.length)];
	}
	
	public static <T> T[] getRandom(final T[] os, final Class<T> c) {
		if (os == null)
			return null;
		if (os.length <= 1)
			return os;
		@SuppressWarnings("unchecked")
		final T[] o = (T[]) Array.newInstance(c, 1);
		o[0] = os[Skript.random.nextInt(os.length)];
		return o;
	}
	
	public static <T> T getRandom(final T[] os, final int start) {
		if (os == null || os.length == 0)
			return null;
		return os[Skript.random.nextInt(os.length - start) + start];
	}
	
	public static <T> T getRandom(final List<T> os) {
		if (os == null || os.size() == 0)
			return null;
		return os.get(Skript.random.nextInt(os.size()));
	}
	
	/**
	 * tests whether two item stacks are of the same type, i.e. it ignores the amounts.
	 * 
	 * @param is1
	 * @param is2
	 * @return
	 */
	public static boolean itemStacksEqual(final ItemStack is1, final ItemStack is2) {
		if (is1 == null || is2 == null)
			return is1 == is2;
		return is1.getTypeId() == is2.getTypeId() && is1.getData().getData() == is2.getData().getData() && is1.getDurability() == is2.getDurability();
	}
	
	/**
	 * note: changes 'from'
	 * 
	 * @param what
	 * @param from
	 * @return from
	 */
	public static final ItemStack remove(final ItemStack what, final ItemStack from) {
		if (what == null || from == null || !itemStacksEqual(what, from))
			return from;
		from.setAmount(Math.max(from.getAmount() - what.getAmount(), 0));
		return from;
	}
	
	/**
	 * note: changes 'to'
	 * 
	 * @param what
	 * @param to
	 * @return to
	 */
	public static final ItemStack add(final ItemStack what, final ItemStack to) {
		if (what == null || to == null || !itemStacksEqual(what, to))
			return to;
		to.setAmount(Math.max(to.getAmount() + what.getAmount(), 0));
		return to;
	}
	
	public static Player getTargetPlayer(final Player player) {
		return getTarget(player, player.getWorld().getPlayers(), Player.class);
	}
	
	public static Entity getTargetEntity(final LivingEntity entity, final Class<? extends Entity> type) {
		if (entity instanceof Creature && !(entity instanceof Player))
			return ((Creature) entity).getTarget();
		return getTarget(entity, entity.getWorld().getEntities(), type);
	}
	
	public static <T extends Entity> T getTarget(final LivingEntity entity, final Iterable<T> entities, final Class<? extends Entity> type) {
		if (entity == null)
			return null;
		T target = null;
		double targetDistanceSquared = Double.MAX_VALUE;
		final double radiusSquared = 1;
		final Vector l = entity.getLocation().toVector();
		l.setY(l.getY() + entity.getEyeHeight());
		final double cos = Math.cos(Math.PI/4);
		for (final T other : entities) {
			if (!(type.isInstance(other)) || other == entity)
				continue;
			if (target == null || targetDistanceSquared > other.getLocation().distanceSquared(entity.getLocation())) {
				final Vector n = other.getLocation().toVector().subtract(l);
				if (entity.getLocation().getDirection().normalize().crossProduct(n).lengthSquared() < radiusSquared && n.normalize().dot(entity.getLocation().getDirection().normalize()) > cos) {
					target = other;
					targetDistanceSquared = target.getLocation().distanceSquared(entity.getLocation());
				}
			}
		}
		return target;
	}
	
	public static int count(final String s, final char c) {
		int r = 0;
		for (final char x : s.toCharArray())
			if (x == c)
				r++;
		return r;
	}
	
}
