/*
 *   This file is part of Skript.
 *
 *  Skript is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Skript is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Skript.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * 
 * Copyright 2011, 2012 Peter Gttinger
 * 
 */

package ch.njol.skript.conditions;

import java.util.List;
import java.util.regex.Matcher;

import org.bukkit.event.Event;

import ch.njol.skript.Skript;
import ch.njol.skript.api.Comparator;
import ch.njol.skript.api.Comparator.ComparatorInfo;
import ch.njol.skript.api.Comparator.Relation;
import ch.njol.skript.api.Condition;
import ch.njol.skript.api.ParseException;
import ch.njol.skript.api.Variable;
import ch.njol.skript.api.intern.ParsedLiteral;

/**
 * @author Peter Gttinger
 * 
 */
public class CondIs extends Condition {
	
	static {
		Skript.addCondition(CondIs.class,
				"%object% ((is|are) (greater|more|higher) than|>) %object%",
				"%object% ((is|are) (less|smaller) than|<) %object%",
				"%object% ((is|are) not|isn't|aren't|!=) (equal to )?%object%",
				"%object% (is|are|=) (equal to )?%object%");
	}
	
	private Variable<?> first, second;
	private Relation relation;
	private Comparator<?, ?> comp;
	private boolean reverseOrder = false;
	
	@Override
	public void init(final List<Variable<?>> vars, final int matchedPattern, final Matcher matcher) throws ParseException {
		first = vars.get(0);
		second = vars.get(1);
		relation = Relation.values()[(matchedPattern + 1) % 4];
		final Class<?> f = first.getReturnType(), s = second.getReturnType();
		Variable<?> temp = null;
		int tempIndex = 0;
		for (final ComparatorInfo<?, ?> info : Skript.getComparators()) {
			
			//TODO comp supports >, < or only =, !=
			
			if (info.c1.isAssignableFrom(f) && info.c2.isAssignableFrom(s)
					|| info.c1.isAssignableFrom(s) && info.c2.isAssignableFrom(f)) {
				comp = info.c;
				reverseOrder = !(info.c1.isAssignableFrom(f) && info.c2.isAssignableFrom(s));
				return;
			} else if (comp != null) {
				for (int c = 0; c <= 1; c++) {
					for (int v = 0; v <= 1; v++) {
						if (info.getType(c).isAssignableFrom((v == 0 ? f : s))) {
							temp = (v == 0 ? first : second).getConvertedVar(info.getType(1 - c));
							if (temp != null) {
								tempIndex = v;
								reverseOrder = c != v;
								comp = info.c;
								c = 1;
								break;
							}
						}
					}
				}
			}
		}
		if (comp != null) {
			if (tempIndex == 0)
				first = temp;
			else
				second = temp;
			return;
		} else {
			for (final ComparatorInfo<?, ?> info : Skript.getComparators()) {
				for (int c = 0; c <= 1; c++) {
					final Variable<?> v1 = first.getConvertedVar(info.getType(c));
					final Variable<?> v2 = second.getConvertedVar(info.getType(1 - c));
					if (v1 != null && v2 != null) {
						first = v1;
						second = v2;
						reverseOrder = c == 1;
						comp = info.c;
						return;
					}
				}
			}
		}
		Variable<?> v1 = first.getConvertedVar(second.getReturnType());
		if (v1 != null) {
			comp = Comparator.equalsComparator;
			first = v1;
			return;
		}
		Variable<?> v2 = second.getConvertedVar(first.getReturnType());
		if (v2 != null) {
			comp = Comparator.equalsComparator;
			second = v2;
			return;
		}
		if (first instanceof ParsedLiteral || second instanceof ParsedLiteral)
			throw new ParseException("the given values are invalid or cannot be compared" + (Skript.logVeryHigh() ? " (" + getDebugMessage(null) + ")" : ""));
		else
			throw new ParseException("the given values cannot be compared" + (Skript.logVeryHigh() ? " (" + getDebugMessage(null) + ")" : ""));
	}
	
	@Override
	public boolean run(final Event e) {
		final Iterable<?> os1 = first.get(e, true);
		final Iterable<?> os2 = second.get(e, true);
		for (final Object o1 : os1) {
			for (final Object o2 : os2) {
				if (!relation.is(reverseOrder ? compare(o2, o1) : compare(o1, o2)))
					return false;
			}
		}
		return true;
	}
	
	@SuppressWarnings("unchecked")
	private <T1, T2> Relation compare(final T1 o1, final T2 o2) {
		return ((Comparator<? super T1, ? super T2>) comp).compare(o1, o2);
	}
	
	@Override
	public String getDebugMessage(final Event e) {
		return first.getDebugMessage(e) + " is " + relation + " " + second.getDebugMessage(e);
	}
	
}
