/*
 *   This file is part of Skript.
 *
 *  Skript is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Skript is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Skript.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * 
 * Copyright 2011, 2012 Peter Gttinger
 * 
 */

package ch.njol.skript;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map.Entry;
import java.util.Random;
import java.util.logging.Level;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.bukkit.Bukkit;
import org.bukkit.ChatColor;
import org.bukkit.command.Command;
import org.bukkit.command.CommandMap;
import org.bukkit.command.CommandSender;
import org.bukkit.event.Event;
import org.bukkit.event.EventHandler;
import org.bukkit.event.EventPriority;
import org.bukkit.event.Listener;
import org.bukkit.event.player.PlayerCommandPreprocessEvent;
import org.bukkit.event.server.ServerCommandEvent;
import org.bukkit.plugin.SimplePluginManager;
import org.bukkit.plugin.java.JavaPlugin;

import ch.njol.skript.api.Comparator;
import ch.njol.skript.api.Comparator.ComparatorInfo;
import ch.njol.skript.api.Comparator.Relation;
import ch.njol.skript.api.Condition;
import ch.njol.skript.api.Converter;
import ch.njol.skript.api.Converter.ConverterInfo;
import ch.njol.skript.api.Converter.ConverterUtils;
import ch.njol.skript.api.Effect;
import ch.njol.skript.api.Expression.Expressions;
import ch.njol.skript.api.Expression.VariableInfo;
import ch.njol.skript.api.Getter;
import ch.njol.skript.api.InitException;
import ch.njol.skript.api.LoopVar;
import ch.njol.skript.api.LoopVar.LoopInfo;
import ch.njol.skript.api.Parser;
import ch.njol.skript.api.SkriptEvent;
import ch.njol.skript.api.SkriptEvent.SkriptEventInfo;
import ch.njol.skript.api.Variable;
import ch.njol.skript.api.intern.ChainedConverter;
import ch.njol.skript.config.Config;
import ch.njol.skript.config.ConfigNode;
import ch.njol.skript.config.EntryNode;
import ch.njol.skript.config.SectionNode;
import ch.njol.skript.data.BukkitEventValues;
import ch.njol.skript.data.DefaultClasses;
import ch.njol.skript.data.DefaultComparators;
import ch.njol.skript.data.DefaultConverters;
import ch.njol.skript.data.SkriptEventValues;
import ch.njol.skript.data.SkriptTriggerItems;
import ch.njol.skript.util.ItemData;
import ch.njol.skript.util.ItemType;
import ch.njol.skript.util.Utils;

/**
 * <b>Skript</b> - A plugin to modify how Minecraft behaves without having to write a single line of code.<br/>
 * <br/>
 * Use this class to extend this plugin's functionality by adding more {@link Condition conditions}, {@link Effect effects}, {@link Variable variables}, etc.<br/>
 * <br/>
 * Please also take a look at the {@link Utils} class! it will probably save you a lot of time.
 * 
 * @author Peter Gttinger
 * 
 */
public class Skript extends JavaPlugin {
	
	// ================ PLUGIN ================
	
	static Skript skript;
	
	public static Skript getPlugin() {
		return skript;
	}
	
	public Skript() {
		skript = this;
		new BukkitEventValues();
		new DefaultClasses();
		new DefaultComparators();
		new DefaultConverters();
		new SkriptTriggerItems();
		new SkriptEventValues();
	}
	
	private static boolean monitorMemoryUsage = false;
	private static long memoryUsed = 0;
	
	@Override
	public void onEnable() {
		System.gc();
		final long startMemory = Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory();
		
		// automatically added by Bukkit now
		// info("loading " + getDescription().getName() + " v" + getDescription().getVersion() + "...");
		
		if (logNormal())
			info(" ~ created &  by Peter Gttinger (aka Njol)");
		loadMainConfig();
		
		monitorMemoryUsage = logHigh();
		
		if (Bukkit.getScheduler().scheduleSyncDelayedTask(this, new Runnable() {
			@Override
			public void run() {
				
				if (monitorMemoryUsage) {
					System.gc();
				}
				final long startMemory = Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory();
				
				loadTriggerFiles();
				info(getDescription().getName() + " finished loading!");
				
				if (monitorMemoryUsage) {
					System.gc();
					final long endMemory = Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory();
					Skript.memoryUsed += endMemory - startMemory;
					info("Skript is currently using " + formatMemory(memoryUsed) + " of RAM");
				}
			}
		}) == -1) {
			error("error scheduling trigger files loader task");
		}
		
		if (monitorMemoryUsage) {
			System.gc();
			final long endMemory = Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory();
			memoryUsed += endMemory - startMemory;
		}
		
	}
	
	private final static String formatMemory(final long memory) {
		double mb = 1. * memory / (1 << 20);
		mb = 1. * Math.round(1000 * mb) / 1000;
		return mb + " MiB";
	}
	
	@Override
	public void onDisable() {
		// info("disabled");
	}
	
	@Override
	public boolean onCommand(final CommandSender sender, final Command command, final String label, final String[] args) {
		return CommandHandler.onCommand(sender, command, label, args);
	}
	
	// ================ CONSTANTS & OTHER ================
	
	public static final String TRIGGERFILEFOLDER = "triggers";
	
	public static final String quotesError = "Invalid use of quotes (\"). If you want to use quotes in \"quoted texts\", double them : \"\".";
	
	public static final double EPSILON = 1e-20;
	
	public static final int MAXBLOCKID = 255;
	
	// TODO option? or in variable?
	public static final int TARGETBLOCKMAXDISTANCE = 100;
	
	public static final Random random = new Random();
	
	private static EventPriority priority = EventPriority.NORMAL;
	
	public static <T> T[] array(final T... array) {
		return array;
	}
	
	// ================ LISTENER FUNCTIONS ================
	
	static boolean listenerEnabled = true;
	
	public static void disableListener() {
		listenerEnabled = false;
	}
	
	public static void enableListener() {
		listenerEnabled = true;
	}
	
	// ================ CONDITIONS & EFFECTS ================
	
	static final ArrayList<VariableInfo<Boolean>> conditions = new ArrayList<VariableInfo<Boolean>>();
	static final ArrayList<VariableInfo<Boolean>> effects = new ArrayList<VariableInfo<Boolean>>();
	static final ArrayList<VariableInfo<Boolean>> topLevelExpressions = new ArrayList<VariableInfo<Boolean>>(20);
	
	/**
	 * registers a {@link Condition}.
	 * 
	 * @param condition
	 * 
	 */
	public static void addCondition(final Class<? extends Condition> condition, final String... patterns) {
		final VariableInfo<Boolean> info = new VariableInfo<Boolean>(patterns, Boolean.class, condition);
		conditions.add(info);
		topLevelExpressions.add(info);
		if (log(Verbosity.VERY_HIGH))
			Skript.info("condition " + condition.getSimpleName() + " added");
	}
	
	/**
	 * registers an {@link Effect}.
	 * 
	 * @param effect
	 * 
	 */
	public static void addEffect(final Class<? extends Effect> effect, final String... patterns) {
		final VariableInfo<Boolean> info = new VariableInfo<Boolean>(patterns, Boolean.class, effect);
		effects.add(info);
		topLevelExpressions.add(info);
		if (log(Verbosity.VERY_HIGH))
			Skript.info("effect " + effect.getSimpleName() + " added");
	}
	
	public static List<VariableInfo<Boolean>> getTopLevelExpressions() {
		return topLevelExpressions;
	}
	
	public static List<VariableInfo<Boolean>> getConditions() {
		return conditions;
	}
	
	public static List<VariableInfo<Boolean>> getEffects() {
		return effects;
	}
	
	// ================ VARIABLES ================
	
	static final ArrayList<VariableInfo<?>> variables = new ArrayList<VariableInfo<?>>(30);
	
	public static <T> void addVariable(final Class<? extends Variable<T>> c, final Class<T> returnType, final String... patterns) {
		variables.add(new VariableInfo<T>(patterns, returnType, c));
		if (log(Verbosity.VERY_HIGH))
			Skript.info("variable " + c.getSimpleName() + " added");
	}
	
	public static List<VariableInfo<?>> getVariables() {
		return variables;
	}
	
	// ================ EVENTS ================
	
	static final ArrayList<SkriptEventInfo> events = new ArrayList<SkriptEventInfo>(50);
	
	@SuppressWarnings("unchecked")
	public static void addEvent(final Class<? extends SkriptEvent> c, final Class<? extends Event> event, final String... patterns) {
		events.add(new SkriptEventInfo(patterns, c, array(event)));
	}
	
	public static void addEvent(final Class<? extends SkriptEvent> c, final Class<? extends Event>[] events, final String... patterns) {
		Skript.events.add(new SkriptEventInfo(patterns, c, events));
	}
	
	public static final ArrayList<SkriptEventInfo> getEvents() {
		return events;
	}
	
	// ================ CONVERTERS ================
	
	private static ArrayList<ConverterInfo<?, ?>> converters = new ArrayList<ConverterInfo<?, ?>>(50);
	
	@SuppressWarnings("unchecked")
	private static <F, M, T> ConverterInfo<F, T> createChainedConverter(final Class<F> from, final Converter<F, M> first, final ConverterInfo<?, ?> second) {
		return new ConverterInfo<F, T>(from, (Class<T>) second.to, new ChainedConverter<F, M, T>(first, (Converter<M, T>) second.converter));
	}
	
	@SuppressWarnings("unchecked")
	private static <F, M, T> ConverterInfo<F, T> createChainedConverter(final ConverterInfo<?, ?> first, final Converter<M, T> second, final Class<T> to) {
		return new ConverterInfo<F, T>((Class<F>) first.from, to, new ChainedConverter<F, M, T>((Converter<F, M>) first.converter, second));
	}
	
	public static <F, T> void addConverter(final Class<F> from, final Class<T> to, final Converter<F, T> converter) {
		final int s = converters.size();
		converters.add(new ConverterInfo<F, T>(from, to, converter));
		for (int i = 0; i < s; i++) {
			final ConverterInfo<?, ?> c = converters.get(i);
			if (to == c.from) {
				converters.add(createChainedConverter(from, converter, c));
			} else if (from == c.to) {
				converters.add(createChainedConverter(c, converter, to));
			}
		}
	}
	
	@SuppressWarnings("unchecked")
	public static <F, T> T convert(final F o, final Class<T> to) {
		for (final ConverterInfo<?, ?> c : converters) {
			if (c.from.isAssignableFrom(o.getClass()) && to.isAssignableFrom(c.to)) {
				final T t = ((Converter<F, T>) c.converter).convert(o);
				if (t != null)
					return t;
			}
		}
		return null;
	}
	
	/**
	 * Tests whether a converter between the given classes exists.
	 * 
	 * @param from
	 * @param to
	 * @return
	 */
	public final static boolean converterExists(final Class<?> from, final Class<?> to) {
		for (final ConverterInfo<?, ?> conv : converters) {
			if (conv.from.isAssignableFrom(from) && to.isAssignableFrom(conv.to))
				return true;
		}
		return false;
	}
	
	public final static <F, T> Converter<? super F, ? extends T> getConverter(final Class<F> from, final Class<T> to) {
		final ConverterInfo<? super F, ? extends T> ci = getConverterInfo(from, to);
		if (ci == null)
			return null;
		return ci.converter;
	}
	
	@SuppressWarnings("unchecked")
	private final static <F, T> ConverterInfo<? super F, ? extends T> getConverterInfo(final Class<F> from, final Class<T> to) {
		for (final ConverterInfo<?, ?> conv : converters) {
			if (conv.from.isAssignableFrom(from) && to.isAssignableFrom(conv.to))
				return (ConverterInfo<? super F, ? extends T>) conv;
		}
		return null;
	}
	
	// ================ CLASSES ================
	
	private static class ClassInfo<T> {
		public ClassInfo(final Class<T> c, final String name, final Class<? extends Variable<T>> defaultVariable, final Parser<T> parser, final String... userInputPatterns) {
			this.c = c;
			this.name = name;
			this.defaultVariable = defaultVariable;
			this.parser = parser;
			this.userInputPatterns = new Pattern[userInputPatterns.length];
			for (int i = 0; i < userInputPatterns.length; i++) {
				this.userInputPatterns[i] = Pattern.compile("^" + userInputPatterns[i] + "$");
			}
		}
		
		Class<T> c;
		String name;
		Class<? extends Variable<T>> defaultVariable;
		Parser<T> parser;
		Pattern[] userInputPatterns;
	}
	
	private final static ArrayList<ClassInfo<?>> classInfos = new ArrayList<Skript.ClassInfo<?>>(50);
	
	public static <T> void addClass(final String name, final Class<T> c, final Class<? extends Variable<T>> defaultVariable, final Parser<T> parser, final String... userInputPatterns) {
		classInfos.add(new ClassInfo<T>(c, name, defaultVariable, parser, userInputPatterns));
	}
	
	private static ClassInfo<?> getClassInfo(final String name) {
		for (final ClassInfo<?> ci : classInfos) {
			if (ci.name.equals(name))
				return ci;
		}
		throw new RuntimeException("no class info found for " + name);
	}
	
	@SuppressWarnings("unchecked")
	private static <T> List<ClassInfo<? extends T>> getClassInfos(final Class<T> c) {
		final ArrayList<ClassInfo<? extends T>> infos = new ArrayList<Skript.ClassInfo<? extends T>>();
		for (final ClassInfo<?> ci : classInfos) {
			if (c.isAssignableFrom(ci.c))
				infos.add((ClassInfo<? extends T>) ci);
		}
		if (infos.size() == 0)
			throw new RuntimeException("no class info found for " + c.getCanonicalName());
		return infos;
	}
	
	public static Class<?> getClass(final String name) {
		return getClassInfo(name).c;
	}
	
	public static Class<?> getClassFromUserInput(final String name) {
		for (final ClassInfo<?> ci : classInfos) {
			for (final Pattern pattern : ci.userInputPatterns) {
				if (pattern.matcher(name).matches())
					return ci.c;
			}
		}
		return null;
	}
	
	public static <T> Class<? extends Variable<?>> getDefaultVariable(final String name) {
		return getClassInfo(name).defaultVariable;
	}
	
	// ======== PARSERS (part of classes) ========
	
	/**
	 * parses without trying to convert anything.
	 * 
	 * @param s
	 * @param c
	 * @return
	 */
	private static <T> T parse_simple(final String s, final Class<T> c) {
		for (final ClassInfo<? extends T> info : getClassInfos(c)) {
			if (info.parser == null)
				continue;
			final T t = info.parser.parse(s);
			if (t != null)
				return t;
		}
		return null;
	}
	
	@SuppressWarnings("unchecked")
	public static <T> T parse(final String s, final Class<T> c) {
		T t = parse_simple(s, c);
		if (t != null)
			return t;
		for (final ConverterInfo<?, ?> conv : converters) {
			if (c.isAssignableFrom(conv.to)) {
				final Object o = parse_simple(s, conv.from);
				if (o != null) {
					t = (T) ConverterUtils.convert(conv, o);
					if (t != null)
						return t;
				}
			}
		}
		return null;
	}
	
	@SuppressWarnings("unchecked")
	public static final <T> Converter<String, ? extends T> getParser(final Class<T> to) {
		for (final ClassInfo<?> ci : classInfos) {
			if (to.isAssignableFrom(ci.c) && ci.parser != null)
				return (Parser<? extends T>) ci.parser;
		}
		for (final ConverterInfo<?, ?> conv : converters) {
			if (to.isAssignableFrom(conv.to)) {
				for (final ClassInfo<?> ci : classInfos) {
					if (conv.from.isAssignableFrom(ci.c) && ci.parser != null)
						return ChainedConverter.newInstance(ci.parser, ci.c, (Converter<?, ? extends T>) conv.converter);
				}
			}
		}
		return null;
	}
	
	/**
	 * @param o any object or array
	 * @return String representation of the object (using a parser if found or String.valueOf(object) otherwise).
	 */
	public static <T> String toString(final T o) {
		return toString(o, false);
	}
	
	@SuppressWarnings("unchecked")
	public static <T> String toString(final T o, final boolean classname) {
		if (o == null)
			return "<none>";
		if (o.getClass().isArray()) {
			final StringBuilder b = new StringBuilder();
			boolean first = true;
			for (final Object i : (Object[]) o) {
				if (!first)
					b.append(", ");
				b.append(toString(i));
				first = false;
			}
			return "[" + b.toString() + "]";
		}
		for (final ClassInfo<?> ci : classInfos) {
			if (ci.parser != null && ci.c.isAssignableFrom(o.getClass())) {
				final String s = ((Parser<T>) ci.parser).toString(o);
				if (s != null)
					return s;
			}
		}
		if (classname)
			return o.getClass().getSimpleName();
		return String.valueOf(o);
	}
	
	// ================ COMPARATORS ================
	
	private final static ArrayList<ComparatorInfo<?, ?>> comparators = new ArrayList<ComparatorInfo<?, ?>>();
	
	public static <T1, T2> void addComparator(final Class<T1> t1, final Class<T2> t2, final Comparator<T1, T2> c) {
		comparators.add(new ComparatorInfo<T1, T2>(t1, t2, c));
	}
	
	public final static List<ComparatorInfo<?, ?>> getComparators() {
		return comparators;
	}
	
	@SuppressWarnings("unchecked")
	public static <T1, T2> Relation compare(final T1 t1, final T2 t2) {
		for (final ComparatorInfo<?, ?> info : comparators) {
			if (info.c1.isInstance(t1) && info.c2.isInstance(t2))
				return ((Comparator<T1, T2>) info.c).compare(t1, t2);
			if (info.c1.isInstance(t2) && info.c2.isInstance(t1))
				return ((Comparator<T2, T1>) info.c).compare(t2, t1);
		}
		if (t1 != null && t1.getClass().isInstance(t2))
			return Relation.get(t1.equals(t2));
		else if (t2 != null && t2.getClass().isInstance(t1))
			return Relation.get(t2.equals(t1));
		return null;
	}
	
	public static Comparator<?, ?> getComparator(final Class<?> c1, final Class<?> c2) {
		if (c1 == null || c2 == null)
			return null;
		for (final ComparatorInfo<?, ?> c : comparators) {
			if (c.c1.isAssignableFrom(c1) && c.c2.isAssignableFrom(c2)
					|| c.c1.isAssignableFrom(c2) && c.c2.isAssignableFrom(c1))
				return c.c;
		}
		if (c1 == c2)
			return Comparator.equalsComparator;
		return null;
	}
	
	// ================ EVENT VALUES ================
	
	static final class EventValueInfo<E extends Event, T> {
		
		Class<E> event;
		Class<T> c;
		Getter<T, E> getter;
		
		public EventValueInfo(final Class<E> event, final Class<T> c, final Getter<T, E> getter) {
			this.event = event;
			this.c = c;
			this.getter = getter;
		}
	}
	
	private static final ArrayList<EventValueInfo<?, ?>> eventValues = new ArrayList<Skript.EventValueInfo<?, ?>>(30);
	
	/**
	 * Adds a value of event.
	 * 
	 * @param e the event type
	 * @param c the type of the default value
	 * @param g the getter to get the value
	 */
	public static <T, E extends Event> void addEventValue(final Class<E> e, final Class<T> c, final Getter<T, E> g) {
		eventValues.add(new EventValueInfo<E, T>(e, c, g));
	}
	
	/**
	 * Adds a value of event. This value will have higher priority than the other.
	 * 
	 * @param e
	 * @param next
	 * @param c
	 * @param g
	 */
	public static <T, E extends Event> void addEventValueBefore(final Class<E> e, final Class<? extends Event> next, final Class<T> c, final Getter<T, E> g) {
		for (int i = 0; i < eventValues.size(); i++) {
			if (eventValues.get(i).event == next) {
				eventValues.add(i, new EventValueInfo<E, T>(e, c, g));
				return;
			}
		}
		eventValues.add(new EventValueInfo<E, T>(e, c, g));
	}
	
	/**
	 * Gets a value assiciated with the event. Returns null if the event doesn't have such a value (conversions are done to try and get the desired value).
	 * 
	 * @param e
	 * @param c
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public static <T, E extends Event> T getEventValue(final E e, final Class<T> c) {
		for (final EventValueInfo<?, ?> ev : eventValues) {
			if (ev.event.isAssignableFrom(e.getClass())) {
				if (c.isAssignableFrom(ev.c)) {
					final T t = ((Getter<? extends T, ? super E>) ev.getter).get(e);
					if (t != null)
						return t;
				} else {
					final ConverterInfo<?, ? extends T> conv = getConverterInfo(ev.c, c);
					if (conv != null) {
						final T t = ConverterUtils.convert(conv, ((Getter<?, ? super E>) ev.getter).get(e));
						if (t != null)
							return t;
					}
				}
			}
		}
		return null;
	}
	
	@SuppressWarnings("unchecked")
	public static final <T, E extends Event> Converter<? super E, ? extends T> getEventValueGetter(final E e, final Class<T> c) {
		for (final EventValueInfo<?, ?> ev : eventValues) {
			if (ev.event.isAssignableFrom(e.getClass())) {
				if (c.isAssignableFrom(ev.c)) {
					return (Getter<? extends T, ? super E>) ev.getter;
				} else {
					return (Converter<? super E, ? extends T>) getConvertedGetter(ev, c);
				}
			}
		}
		return null;
	}
	
	private final static <E extends Event, F, T> Converter<? super E, ? extends T> getConvertedGetter(final EventValueInfo<E, F> i, final Class<T> to) {
		final Converter<? super F, ? extends T> c = getConverter(i.c, to);
		if (c == null)
			return null;
		return new Converter<E, T>() {
			@Override
			public T convert(final E e) {
				return c.convert(i.getter.get(e));
			}
		};
	}
	
	// ================ LOOPS ================
	
	static final ArrayList<LoopInfo<?>> loops = new ArrayList<LoopVar.LoopInfo<?>>();
	
	public static <T> void addLoop(final Class<? extends LoopVar<T>> c, final Class<T> returnType, final String var, final String... patterns) {
		loops.add(new LoopInfo<T>(c, returnType, var, patterns));
	}
	
	public static String getLoopVarName(final Class<? extends LoopVar<?>> c) {
		for (final LoopInfo<?> info : loops) {
			if (info.c.equals(c))
				return info.var;
		}
		throw new RuntimeException("internal error");
	}
	
	// ================ COMMANDS ================
	
	private static final ArrayList<SkriptCommand> commands = new ArrayList<SkriptCommand>();
	
	static CommandMap commandMap = null;
	static {
		try {
			
			if (Bukkit.getPluginManager() instanceof SimplePluginManager) {
				final Field f = SimplePluginManager.class.getDeclaredField("commandMap");
				f.setAccessible(true);
				
				commandMap = (CommandMap) f.get(Bukkit.getPluginManager());
			}
			
		} catch (final NoSuchFieldException e) {
			outdatedError(e);
		} catch (final SecurityException e) {
			outdatedError(e);
		} catch (final IllegalArgumentException e) {
			outdatedError(e);
		} catch (final IllegalAccessException e) {
			outdatedError(e);
		}
	}
	
	public static void addCommand(final SkriptCommand command) {
		commands.add(command);
		commandMap.register("/", command);
	}
	
	public static void outdatedError() {
		error("The version of Skript you're using is not compatible with " + Bukkit.getVersion());
	}
	
	public static void outdatedError(final Throwable t) {
		outdatedError();
		t.printStackTrace();
	}
	
	// ================ LOGGING ================
	
	public static final boolean logNormal() {
		return SkriptLogger.log(Verbosity.NORMAL);
	}
	
	public static final boolean logHigh() {
		return SkriptLogger.log(Verbosity.HIGH);
	}
	
	public static final boolean logVeryHigh() {
		return SkriptLogger.log(Verbosity.VERY_HIGH);
	}
	
	public static final boolean logExtreme() {
		return SkriptLogger.log(Verbosity.EXTREME);
	}
	
	public static final boolean log(final Verbosity minVerb) {
		return SkriptLogger.log(minVerb);
	}
	
	public static void config(final String info) {
		SkriptLogger.log(Level.CONFIG, info);
	}
	
	public static void info(final String info) {
		SkriptLogger.log(Level.INFO, info);
	}
	
	public static void warning(final String warning) {
		SkriptLogger.log(Level.WARNING, warning);
	}
	
	private static String cause = "";
	
	public static void printErrorAndCause(final String error) {
		SkriptLogger.log(Level.SEVERE, error + (hasWithheldError() ? " because " + cause : ""));
		clearErrorCause();
	}
	
	public static void printErrorCause(final CommandSender sender) {
		if (sender != null)
			sender.sendMessage(cause);
		else
			SkriptLogger.log(Level.SEVERE, cause);
		clearErrorCause();
	}
	
	public static void printErrorCause() {
		SkriptLogger.log(Level.SEVERE, cause);
		clearErrorCause();
	}
	
	/**
	 * Should not be used except in the {@link InitException} catch clause of {@link Expressions#parse(String, java.util.Iterator)}, as well as bythe tigger file loader.
	 */
	public static final void clearErrorCause() {
		cause = "";
	}
	
	public static final boolean hasWithheldError() {
		return !cause.isEmpty();
	}
	
	public static void setErrorCause(final String error, final boolean overwrite) {
		if (overwrite || !hasWithheldError())
			cause = error;
	}
	
	public static void error(final String error) {
		SkriptLogger.log(Level.SEVERE, error);
	}
	
	// ================ ALIASES ================
	
	/**
	 * Do not use/change this.
	 */
	public static final HashMap<String, ItemType> aliases = new HashMap<String, ItemType>(1500, 0.5f);
	public static final ItemType everything = new ItemType();
	static {
		everything.all = true;
		everything.add(new ItemData());
		// this is not an alias!
	}
	
	private static HashMap<String, ItemType> getAliases(final String name, final ItemType value, final HashMap<String, HashMap<String, ItemType>> variations) {
		final HashMap<String, ItemType> r = new HashMap<String, ItemType>();
		Matcher m;
		if ((m = Pattern.compile("\\[(.+?)\\]").matcher(name)).find()) {
			r.putAll(getAliases(m.replaceFirst(""), value, variations));
			r.putAll(getAliases(m.replaceFirst("$1"), value, variations));
		} else if ((m = Pattern.compile("\\((.+?)\\)").matcher(name)).find()) {
			final String[] split = m.group(1).split("\\|");
			if (split.length == 1) {
				error("brackets have a special meaning in aliases and cannot be used as usual");
			}
			for (final String s : split) {
				r.putAll(getAliases(m.replaceFirst(s), value, variations));
			}
		} else if ((m = Pattern.compile("\\{(.+?)\\}").matcher(name)).find()) {
			if (variations.get(m.group(1)) != null) {
				boolean hasDefault = false;
				for (final Entry<String, ItemType> v : variations.get(m.group(1)).entrySet()) {
					if (v.getKey().equalsIgnoreCase("{default}")) {
						hasDefault = true;
						final String n = m.replaceFirst("");
						r.putAll(getAliases(n, addInfo(v.getValue().intersection(value), n), variations));
					} else {
						final String n = m.replaceFirst(v.getKey());
						r.putAll(getAliases(n, addInfo(v.getValue().intersection(value), n), variations));
					}
				}
				if (!hasDefault)
					r.putAll(getAliases(m.replaceFirst(""), value, variations));
			} else {
				error("unknown variation {" + m.group(1) + "}");
			}
		} else {
			r.put(name, addInfo(value, name));
		}
		return r;
	}
	
	private static ItemType addInfo(final ItemType t, final String n) {
		ItemType i;
		if (n.endsWith(" block") && (i = aliases.get(n.substring(0, n.length() - " block".length()))) != null) {
			i.setBlock(t);
		} else if (n.endsWith(" item") && (i = aliases.get(n.substring(0, n.length() - " item".length()))) != null) {
			i.setItem(t);
		} else if ((i = aliases.get(n + " item")) != null) {
			t.setItem(i);
		} else if ((i = aliases.get(n + " block")) != null) {
			t.setBlock(i);
		}
		return t;
	}
	
	private static int addAliases(final String name, final String value, final HashMap<String, HashMap<String, ItemType>> variations) {
		final ItemType t = Utils.parseAlias(value);
		if (t == null) {
			Skript.printErrorAndCause("'" + value + "' is invalid");
			return 0;
		}
		final HashMap<String, ItemType> as = getAliases(name, t, variations);
		boolean printedStartingWithNumberError = false;
		boolean printedSyntaxError = false;
		for (final Entry<String, ItemType> e : as.entrySet()) {
			final String s = e.getKey().trim().replaceAll("\\s+", " ").toLowerCase(Locale.ENGLISH);
			if (s.matches("\\d+ .*")) {
				if (!printedStartingWithNumberError) {
					error("aliases must not start with a number");
					printedStartingWithNumberError = true;
				}
				continue;
			}
			if (s.contains(",") || s.contains(" and ") || s.contains(" or ")) {
				if (!printedSyntaxError) {
					error("aliases must not contain syntax elements (comma, 'and', 'or')");
					printedSyntaxError = true;
				}
				continue;
			}
			aliases.put(s, e.getValue());
//			if (logSpam()) <- =P
//				info("added alias " + s + " for " + e.getValue());
		}
		return as.size();
	}
	
	// ================ CONFIGS ================
	
	static Config mainConfig;
	static final ArrayList<Config> configs = new ArrayList<Config>();
	
	private static boolean keepConfigsLoaded = true;
	private static boolean enableEffectCommands = false;
	
	private static final void parseMainConfig() throws IOException {
		
		String[] aliasNodes = null;
		
		SkriptLogger.verbosity = null;
		priority = null;
		boolean isKeepConfigsLoadedSet = false;
		boolean isEnableEffectCommandsSet = false;
		
		for (final ConfigNode node : mainConfig.getMainNode()) {
			if (node.isEntry()) {
				if (((EntryNode) node).getKey().equalsIgnoreCase("verbosity")) {
					try {
						SkriptLogger.verbosity = Verbosity.valueOf(((EntryNode) node).getValue().toUpperCase(Locale.ENGLISH).replace(' ', '_'));
					} catch (final IllegalArgumentException e) {
						Skript.error("Invalid verbosity. Allowed values: low, normal, high, very high, extreme");
					}
				} else if (((EntryNode) node).getKey().equalsIgnoreCase("plugin priority")) {
					try {
						priority = EventPriority.valueOf(((EntryNode) node).getValue().toUpperCase(Locale.ENGLISH));
					} catch (final IllegalArgumentException e) {
						Skript.error("Invalid plugin priority. Allowed values: lowest, low, normal, high, highest");
					}
				} else if (((EntryNode) node).getKey().equalsIgnoreCase("aliases")) {
					aliasNodes = ((EntryNode) node).getValue().split(",");
					for (int i = 0; i < aliasNodes.length; i++) {
						aliasNodes[i] = aliasNodes[i].trim();
					}
				} else if (((EntryNode) node).getKey().equalsIgnoreCase("keep configs loaded")) {
					isKeepConfigsLoadedSet = true;
					keepConfigsLoaded = Utils.parseBoolean(((EntryNode) node).getValue());
				} else if (((EntryNode) node).getKey().equalsIgnoreCase("enable effect commands")) {
					isEnableEffectCommandsSet = true;
					enableEffectCommands = Utils.parseBoolean(((EntryNode) node).getValue());
				} else {
					Skript.error("unknown option");
				}
			} else if (node.isSection()) {
				if (Utils.contains(aliasNodes, node.getName()) == -1) {
					Skript.error("Invalid section. If this is an alias section add it to 'aliases' so it will be loaded.");
				}
			}
		}
		
		// if I have to add about two more options, I'll create my own config API
		if (SkriptLogger.verbosity == null) {
			warning("config.cfg: 'verbosity' is missing, defaulting to normal");
			SkriptLogger.verbosity = Verbosity.NORMAL;
		}
		if (priority == null) {
			warning("config.cfg: 'plugin priority' is missing, defaulting to normal");
			priority = EventPriority.NORMAL;
		}
		if (!isKeepConfigsLoadedSet) {
			warning("config.cfg: 'keep configs loaded' is missing, defaulting to true");
			keepConfigsLoaded = true;
		}
		if (!isEnableEffectCommandsSet) {
			warning("config.cfg: 'enable effect commands' is missing, defaulting to false");
			enableEffectCommands = false;
		}
		
		if (aliasNodes == null) {
			warning("config.cfg: option 'aliases' is missing, no aliases will be loaded");
		} else {
			final HashMap<String, HashMap<String, ItemType>> variations = new HashMap<String, HashMap<String, ItemType>>();
			int num = 0;
			for (final String an : aliasNodes) {
				final ConfigNode node = mainConfig.getMainNode().get(an);
				SkriptLogger.setNode(node);
				if (node == null) {
					error("alias section '" + an + "' not found!");
					continue;
				}
				if (!node.isSection()) {
					error("aliases have to be in sections, but '" + an + "' is not a section!");
					continue;
				}
				int i = 0;
				for (final ConfigNode n : (SectionNode) node) {
					if (n.isEntry()) {
						i += addAliases(((EntryNode) n).getKey(), ((EntryNode) n).getValue(), variations);
					} else if (n.isSection()) {
						if (!(n.getName().startsWith("{") && n.getName().endsWith("}"))) {
							Skript.error("unexpected non-variation section");
							continue;
						}
						final HashMap<String, ItemType> vs = new HashMap<String, ItemType>();
						for (final ConfigNode a : (SectionNode) n) {
							if (a.isVoid())
								continue;
							if (a.isSection()) {
								Skript.error("unexpected section");
								continue;
							}
							final ItemType t = Utils.parseAlias(((EntryNode) a).getValue());
							if (t != null)
								vs.put(((EntryNode) a).getKey(), t);
							else
								Skript.printErrorAndCause("'" + ((EntryNode) a).getValue() + "' is invalid");
						}
						variations.put(n.getName().substring(1, n.getName().length() - 1), vs);
					}
				}
				if (log(Verbosity.HIGH))
					info("loaded " + i + " aliases from " + node.getName());
				num += i;
			}
			SkriptLogger.setNode(null);
			
			if (!keepConfigsLoaded)
				mainConfig = null;
			
			if (log(Verbosity.NORMAL))
				info("loaded a total of " + num + " aliases");
		}
		
		if (enableEffectCommands) {
			if (commandMap == null) {
				error("you have to use CraftBukkit to use commands");
			} else {
				Bukkit.getPluginManager().registerEvents(new Listener() {
					@SuppressWarnings("unused")
					@EventHandler(priority = EventPriority.HIGHEST, ignoreCancelled = true)
					public void onPlayerCommand(final PlayerCommandPreprocessEvent e) {
						if (Skript.handleEffectCommand(e.getPlayer(), e.getMessage().substring(1)))
							e.setCancelled(true);
					}
					
					@SuppressWarnings("unused")
					@EventHandler(priority = EventPriority.HIGHEST)
					public void onServerCommand(final ServerCommandEvent e) {
						if (Skript.handleEffectCommand(e.getSender(), e.getCommand()))
							e.setCommand("");
					}
				}, skript);
			}
		}
	}
	
	/**
	 * only called if {@link Skript#enableEffectCommands} is true
	 * 
	 * @param sender
	 * @param command
	 */
	private static boolean handleEffectCommand(final CommandSender sender, final String command) {
		if (!sender.hasPermission("skript.commands"))
			return false;
		final String x = command.split(" ", 2)[0];
		if (commandMap.getCommand(x) != null)
			return false;
		
		final Effect e = Effect.parse(command);
		if (e != null) {
			final String[] c = command.split(" ");
			sender.sendMessage(ChatColor.GRAY + ChatColor.stripColor(command));
			e.run(new CommandEvent(sender, c[0], c.length > 1 ? Arrays.copyOfRange(c, 1, c.length - 1) : new String[0]));
			return true;
		} else if (Skript.hasWithheldError()) {
			sender.sendMessage(ChatColor.GRAY + ChatColor.stripColor(command));
			Skript.printErrorCause(sender);
			return true;
		}
		return false;
	}
	
	void loadMainConfig() {
		if (logNormal())
			info("loading main config...");
		boolean successful = true;
		try {
			
			final File config = new File(getDataFolder(), "config.cfg");
			if (!config.exists() || !config.canRead()) {
				error("Config file 'config.cfg' does not exist or is not accessible!");
				return;
			}
			mainConfig = new Config(config, false);
			parseMainConfig();
			
		} catch (final Exception e) {
			error("error loading config:");
			e.printStackTrace();
			successful = false;
		}
		if (successful && logNormal())
			info("main config loaded successfully.");
	}
	
	void loadTriggerFiles() {
		boolean successful = true;
		int numFiles = 0;
		int numTriggers = 0;
		try {
			final File includes = new File(getDataFolder(), Skript.TRIGGERFILEFOLDER + File.separatorChar);
			for (final File f : includes.listFiles(new FilenameFilter() {
				@Override
				public boolean accept(final File dir, final String name) {
					return name.endsWith(".cfg") && !name.startsWith("-");
				}
			})) {
				final Config c = new Config(f, true);
				if (keepConfigsLoaded)
					configs.add(c);
				numTriggers += TriggerFileLoader.load(c);
				numFiles++;
			}
		} catch (final Exception e) {
			SkriptLogger.setNode(null);
			error("error loading trigger files:");
			e.printStackTrace();
			successful = false;
		}
		if (successful && log(Verbosity.NORMAL))
			info("loaded " + numFiles + " trigger file" + (numFiles == 1 ? "" : "s") + " with a total of " + numTriggers + " trigger" + (numTriggers == 1 ? "" : "s"));
		for (final Entry<Class<? extends Event>, List<Trigger>> e : SkriptEventHandler.triggers.entrySet()) {
			Bukkit.getServer().getPluginManager().registerEvent(e.getKey(), new Listener() {}, priority, SkriptEventHandler.ee, skript);
			if (log(Verbosity.EXTREME))
				config("registered Event " + e.getKey().getSimpleName());
		}
	}
	
}
