/*
 * This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 2.5 Switzerland License. To view a copy of this license, visit
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ch/ or send a letter to Creative Commons, 444 Castro Street, Suite 900, Mountain View, California, 94041, USA.
 */

package ch.njol.skript.variables;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;

import org.bukkit.block.Block;
import org.bukkit.entity.Creature;
import org.bukkit.entity.Entity;
import org.bukkit.entity.LivingEntity;
import org.bukkit.event.Event;

import ch.njol.skript.Skript;
import ch.njol.skript.api.Changer.ChangeMode;
import ch.njol.skript.api.Variable;
import ch.njol.skript.api.intern.ConvertedVariable;
import ch.njol.skript.data.DefaultChangers;
import ch.njol.skript.util.EntityType;
import ch.njol.skript.util.Utils;

/**
 * @author Peter Gttinger
 * 
 */
public class VarTarget extends Variable<Object> {
	
	static {
		Skript.addVariable(VarTarget.class, Object.class, "targeted block( of %livingentity%)?",
				"target(ed %entitytype%)( of %livingentity%)?");
	}
	
	private Variable<EntityType> types = null;
	private Variable<LivingEntity> entities;
	
	@SuppressWarnings("unchecked")
	@Override
	public void init(final List<Variable<?>> vars, final int matchedPattern, final Matcher matcher) {
		if (vars.size() == 1)
			types = (Variable<EntityType>) vars.get(0);
		entities = (Variable<LivingEntity>) vars.get(vars.size() - 1);
	}
	
	@Override
	protected Object[] getAll(final Event evt) {
		if (types != null) {
			final ArrayList<Entity> targets = new ArrayList<Entity>();
			final EntityType[] types = this.types.get(evt);
			for (final LivingEntity e : entities.get(evt)) {
				for (final EntityType type : types) {
					final Entity t = Utils.getTargetEntity(e, type.c);
					if (t != null)
						targets.add(t);
				}
			}
			return targets.toArray(new Entity[0]);
		} else {
			final ArrayList<Block> targets = new ArrayList<Block>();
			for (final LivingEntity e : entities.get(evt)) {
				final Block t = e.getTargetBlock(null, Skript.TARGETBLOCKMAXDISTANCE);
				if (t != null)
					targets.add(t);
			}
			return targets.toArray(new Block[0]);
		}
	}
	
	@Override
	public <R> ConvertedVariable<?, R> getConvertedVar(final Class<R> to) {
		if (types != null) {
			return ConvertedVariable.newInstance(this, Entity.class, to);
		} else {
			return ConvertedVariable.newInstance(this, Block.class, to);
		}
	}
	
	@Override
	public Class<?> getReturnType() {
		if (types != null)
			return Entity.class;
		else
			return Block.class;
	}
	
	@Override
	public String getDebugMessage(final Event e) {
		if (e == null)
			return types == null ? "targeted block of " + entities.getDebugMessage(e) : "targeted " + types.getDebugMessage(e) + " of " + entities.getDebugMessage(e);
		return Skript.toString(getAll(e));
	}
	
	@Override
	public Class<?> acceptChange(final ChangeMode mode) {
		if (types == null)
			return DefaultChangers.blockChanger.acceptChange(mode);
		if (mode == ChangeMode.CLEAR || mode == ChangeMode.SET)
			return LivingEntity.class;
		return null;
	}
	
	@Override
	public void change(final Event e, final Object[] delta, final ChangeMode mode) {
		if (types == null) {
			DefaultChangers.blockChanger.change((Block[]) get(e), delta, mode);
		} else {
			for (final LivingEntity entity : entities.get(e)) {
				if (!(entity instanceof Creature))
					continue;
				((Creature) entity).setTarget((LivingEntity) delta[0]);
			}
		}
	}
	
}
