/*
 * This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 2.5 Switzerland License. To view a copy of this license, visit
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ch/ or send a letter to Creative Commons, 444 Castro Street, Suite 900, Mountain View, California, 94041, USA.
 */

package ch.njol.skript.util;

import java.util.ArrayList;

import org.bukkit.event.Event;

import ch.njol.skript.Skript;
import ch.njol.skript.api.Debuggable;
import ch.njol.skript.api.Variable;

/**
 * 
 * represents a string that may contain variables.
 * 
 * @author Peter Gttinger
 * 
 */
public class VariableString implements Debuggable {
	private final ArrayList<Object> string = new ArrayList<Object>();
	private Event last = null;
	private String lastString = null;
	private final boolean isSimple;
	
	public VariableString(final String s) {
		if (!s.contains("%")) {
			lastString = s;
			isSimple = true;
			return;
		}
		int c = s.indexOf('%');
		string.add(s.substring(0, c));
		while (c != s.length()) {
			final int c2 = s.indexOf('%', c + 1);
			if (c2 == -1) {
				Skript.error("The percent sign is used for variables (e.g. %player%). To insert a %, type it twice: %%. (found in \"" + s + "\")");
				isSimple = true;
				return;
			}
			int p = s.indexOf('(', c + 1);
			String[] params = new String[0];
			if (p != -1 && p < c2) {
				params = s.substring(p + 1, s.indexOf(')', p + 1)).split(", ?");
			}
			if (c + 1 == c2) {
				string.add("%");
			} else {
				if (params.length == 0)
					p = c2;
				final Variable<?> var = Variable.parse(s.substring(c + 1, p), Object.class);
				if (var == null) {
					Skript.printErrorAndCause("can't understand the variable %" + s.substring(c + 1, c2) + "%");
					isSimple = true;
					return;
				} else {
					string.add(var);
				}
			}
			c = s.indexOf('%', c2 + 1);
			if (c == -1)
				c = s.length();
			string.add(s.substring(c2 + 1, c));
		}
		isSimple = false;
	}
	
	public static VariableString[] makeStrings(final String[] args) {
		final VariableString[] strings = new VariableString[args.length];
		for (int i = 0; i < args.length; i++) {
			strings[i] = new VariableString(args[i]);
		}
		return strings;
	}
	
	/**
	 * Parses all variables in the string and returns it. The returned string is cached as long as this method is always called with the same event argument.
	 * 
	 * @param e Event to pass to the variables.
	 * @return The input string with all variables replaced.
	 */
	public String get(final Event e) {
		if (isSimple || last == e)
			return lastString;
		final StringBuilder b = new StringBuilder();
		for (final Object o : string) {
			if (o instanceof Variable) {
				b.append(Skript.toString(((Variable<?>) o).get(e)));
			} else if (o instanceof String) {
				b.append((String) o);
			}
		}
		last = e;
		return lastString = b.toString();
	}
	
	@Override
	public String getDebugMessage(final Event e) {
		if (isSimple)
			return '"' + lastString + '"';
		final StringBuilder b = new StringBuilder();
		b.append('"');
		for (final Object o : string) {
			if (o instanceof Variable) {
				b.append("%" + ((Variable<?>) o).getDebugMessage(e) + "%");
			} else {
				b.append("" + o);
			}
		}
		b.append('"');
		return b.toString();
	}
}
