/*
 * This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 2.5 Switzerland License. To view a copy of this license, visit
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ch/ or send a letter to Creative Commons, 444 Castro Street, Suite 900, Mountain View, California, 94041, USA.
 */

package ch.njol.skript.util;

import java.util.Arrays;

import org.bukkit.Location;
import org.bukkit.block.Block;
import org.bukkit.block.BlockFace;

import ch.njol.skript.Skript;

/**
 * 
 * @author Peter Gttinger
 * 
 */
public class Offset {
	private double[] mod = {0, 0, 0};
	private boolean isOffset;
	
	public Offset(final String[] args) {
		if (args.length == 3) {
			try {
				mod[0] = Double.parseDouble(args[0]);
				mod[1] = Double.parseDouble(args[1]);
				mod[2] = Double.parseDouble(args[2]);
				checkIsOffset();
				return;
			} catch (final NumberFormatException e) {}
		}
		for (final String arg : args) {
			boolean ok = true;
			final BlockFace f = Utils.getBlockFace(arg.replaceFirst("^[0-9.]+ ?", ""), false);
			if (f != null) {
				try {
					final double a = Double.parseDouble(arg.replaceFirst("[^0-9.].*$", ""));
					mod[0] += f.getModX() * a;
					mod[1] += f.getModY() * a;
					mod[2] += f.getModZ() * a;
				} catch (final NumberFormatException e) {
					ok = false;
				}
			} else {
				ok = false;
			}
			if (!ok) {
				Skript.setErrorCause("'" + arg + "' is not an offset", true);
				isOffset = false;
				return;
			}
			
		}
		checkIsOffset();
	}
	
	private void checkIsOffset() {
		isOffset = mod[0] != 0 || mod[1] != 0 || mod[2] != 0;
	}
	
	public Offset(final double modX, final double modY, final double modZ) {
		mod[0] = modX;
		mod[1] = modY;
		mod[2] = modZ;
		checkIsOffset();
	}
	
	public Offset(final double[] mod) {
		if (mod == null || mod.length != 3)
			throw new IllegalArgumentException();
		this.mod = mod;
		checkIsOffset();
	}
	
	public Block setOff(final Block block) {
		if (block == null || !isOffset)
			return block;
		return block.getRelative((int) Math.round(mod[0]), (int) Math.round(mod[1]), (int) Math.round(mod[2]));
	}
	
	public Location setOff(final Location location) {
		if (location == null || !isOffset)
			return location;
		return new Location(location.getWorld(), location.getX() + mod[0], location.getY() + mod[1], location.getZ() + mod[2]);
	}
	
	public boolean isOffset() {
		return isOffset;
	}
	
	/**
	 * @return "(dx,dy,dz)"
	 */
	@Override
	public String toString() {
		return "(" + mod[0] + "," + mod[1] + "," + mod[2] + ")";
	}
	
	@Override
	public int hashCode() {
		return Arrays.hashCode(mod);
	}
	
	@Override
	public boolean equals(final Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final Offset other = (Offset) obj;
		if (!Arrays.equals(mod, other.mod))
			return false;
		return true;
	}
	
	public static Offset combine(final Offset[] offsets) {
		if (offsets == null || offsets.length == 0)
			return null;
		if (offsets.length == 1)
			return offsets[0];
		final double[] mod = new double[3];
		for (final Offset o : offsets) {
			for (int axis = 0; axis < 3; axis++) {
				mod[axis] += o.mod[axis];
			}
		}
		return new Offset(mod);
	}
	
}
