/*
 * This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 2.5 Switzerland License. To view a copy of this license, visit
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ch/ or send a letter to Creative Commons, 444 Castro Street, Suite 900, Mountain View, California, 94041, USA.
 */

package ch.njol.skript.util;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;

import org.bukkit.block.Block;
import org.bukkit.inventory.Inventory;
import org.bukkit.inventory.ItemStack;

import ch.njol.skript.Skript;

public class ItemType implements Cloneable {
	
	public ArrayList<ItemData> types = new ArrayList<ItemData>();
	
	private ItemType item = null;
	private ItemType block = null;
	
	public ItemType() {}
	
	public ItemType(final ItemData... types) {
		for (final ItemData type : types) {
			this.types.add(type);
		}
	}
	
	public ItemType(final ArrayList<ItemData> types) {
		this.types = types;
	}
	
	public ItemType(final ItemStack i) {
		types.add(new ItemData(i));
	}
	
	public ItemType(final Block block) {
		final ItemData d = new ItemData();
		d.typeid = block.getTypeId();
		d.dataMin = d.dataMax = block.getData();
		types.add(d);
	}
	
	public void setItem(final ItemType item) {
		this.item = item;
	}
	
	public void setBlock(final ItemType block) {
		this.block = block;
	}
	
	public ItemType getItem() {
		return item == null ? this : item;
	}
	
	public ItemType getBlock() {
		return block == null ? this : block;
	}
	
	public boolean isOfType(final ItemStack item) {
		if (item == null)
			return false;
		for (final ItemData type : types) {
			if (type.isOfType(item))
				return true;
		}
		return false;
	}
	
	public boolean isOfType(final Block block) {
		for (final ItemData d : types) {
			if ((d.typeid == -1 || block.getTypeId() == d.typeid) && (d.dataMin == -1 || block.getData() >= d.dataMin) && (d.dataMax == -1 || block.getData() <= d.dataMax))
				return true;
		}
		return false;
	}
	
	/**
	 * @return "(itemdata, ...)"
	 */
	@Override
	public String toString() {
		return "("+Utils.join(types)+")";
	}
	
	public boolean setBlock(final Block block, final boolean applyPhysics) {
		final ItemType b = getBlock();
		ItemData d;
		final boolean[] tried = new boolean[b.types.size()];
		while (true) {
			final int i = Skript.random.nextInt(types.size());
			if ((d = b.types.get(i)).typeid < 256)
				break;
			for (final boolean t : tried) {
				if (!t)
					continue;
			}
			return false;
		}
		if (d.typeid != -1)
			block.setTypeId(d.typeid, applyPhysics && d.dataMin == -1 && d.dataMax == -1);
		else
			block.setTypeId(Skript.random.nextInt(256), applyPhysics && d.dataMin == -1 && d.dataMax == -1);
		if (d.dataMin != -1 || d.dataMax != -1) {
			block.setData((byte) d.getData(), applyPhysics);
		}
		return true;
	}
	
	/**
	 * intersects all ItemDatas with all ItemDatas of the given ItemType, returning an ItemType with n*m ItemDatas, where n = #ItemDatas of this ItemType, and m = #ItemDatas of the
	 * argument.
	 * 
	 * more info: {@link ItemData#intersection(ItemData)}
	 * 
	 * @param other
	 * @return
	 */
	public ItemType intersection(final ItemType other) {
		final ItemType r = new ItemType();
		for (final ItemData d1 : types) {
			for (final ItemData d2 : other.types) {
				r.types.add(d1.intersection(d2));
			}
		}
		return r;
	}
	
	/**
	 * 
	 * @return a random ItemStack of this type.
	 */
	public ItemStack getRandom() {
		return Utils.getRandom(getItem().types).getRandom();
	}
	
	public Iterator<ItemStack> getAll() {
		return new Iterator<ItemStack>() {
			
			ListIterator<ItemData> iter = types.listIterator();
			Iterator<ItemStack> currentDataIter;
			
			@Override
			public boolean hasNext() {
				return iter.hasNext() || currentDataIter.hasNext();
			}
			
			@Override
			public ItemStack next() {
				if (currentDataIter == null || !currentDataIter.hasNext()) {
					currentDataIter = iter.next().getAll();
				}
				return currentDataIter.next();
			}
			
			@Override
			public void remove() {}
			
		};
	}
	
	/**
	 * removes this type from the item stack if applicable
	 * 
	 * @param item
	 * @return the passed ItemStack
	 */
	public ItemStack removeFrom(final ItemStack item) {
		if (item == null)
			return null;
		for (final ItemData type : types) {
			if (type.isOfType(item)) {
				item.setAmount(Math.max(item.getAmount() - (type.amount == -1 ? 1 : type.amount), 0));
				return item;
			}
		}
		return item;
	}
	
	public ItemStack addTo(final ItemStack item) {
		if (item == null)
			return null;
		for (final ItemData type : types) {
			if (type.isOfType(item)) {
				item.setAmount(Math.max(item.getAmount() + (type.amount == -1 ? 1 : type.amount), 0));
				return item;
			}
		}
		return item;
	}
	
	public boolean removeFrom(final Inventory invi) {
		final ItemStack is = getRandom();
		for (int i = 0; i < invi.getSize(); i++) {
			final ItemStack is2 = invi.getItem(i);
			if (Utils.itemStacksEqual(is, is2)) {
				final int d = Math.min(is2.getAmount(), is.getAmount());
				if (d > 0) {
					is.setAmount(is.getAmount() - d);
					is2.setAmount(is2.getAmount() - d);
					invi.setItem(i, is2);
					if (is.getAmount() == 0)
						return true;
				}
			}
		}
		return false;
	}
	
	public boolean removeFrom(final List<ItemStack> list) {
		final ItemStack is = getRandom();
		for (int i = 0; i < list.size(); i++) {
			final ItemStack is2 = list.get(i);
			if (Utils.itemStacksEqual(is, is2)) {
				final int d = Math.min(is2.getAmount(), is.getAmount());
				if (d > 0) {
					is.setAmount(is.getAmount() - d);
					is2.setAmount(is2.getAmount() - d);
					list.set(i, is2);
					if (is.getAmount() == 0)
						return true;
				}
			}
		}
		return false;
	}
	
	@Override
	public ItemType clone() {
		final ArrayList<ItemData> datas = new ArrayList<ItemData>(types.size());
		for (final ItemData d : types)
			datas.add(d.clone());
		return new ItemType(datas);
	}
	
}
