/*
 * This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 2.5 Switzerland License. To view a copy of this license, visit
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ch/ or send a letter to Creative Commons, 444 Castro Street, Suite 900, Mountain View, California, 94041, USA.
 */

package ch.njol.skript.util;

import java.util.Arrays;
import java.util.Iterator;

import org.bukkit.Material;
import org.bukkit.inventory.ItemStack;

import ch.njol.skript.Skript;
import ch.njol.util.SingleItemIterator;

/**
 * 
 * @author Peter Gttinger
 * 
 */
public class ItemData implements Cloneable {
	public int typeid = -1;
	public short dataMin = -1;
	public short dataMax = -1;
	public int amount = -1;
	
	public ItemData(final int typeid, final short d, final int amount) {
		this.typeid = typeid;
		dataMin = dataMax = d;
		this.amount = amount;
	}
	
	public ItemData(final int typeid, final short dMin, final short dMax, final int amount) {
		this.typeid = typeid;
		dataMin = dMin;
		dataMax = dMax;
		this.amount = amount;
	}
	
	public ItemData() {
		typeid = -1;
		dataMin = -1;
		dataMax = -1;
		amount = -1;
	}
	
	public ItemData(final ItemStack i) {
		typeid = i.getTypeId();
		dataMin = dataMax = i.getDurability() != 0 ? i.getDurability() : i.getData().getData();
		amount = i.getAmount();
	}
	
	public ItemData(final ItemData other) {
		typeid = other.typeid;
		amount = other.amount;
		dataMax = other.dataMax;
		dataMin = other.dataMin;
	}
	
	public void merge(final ItemData other) {
		if (other == null)
			return;
		if (other.typeid != -1)
			typeid = other.typeid;
		if (other.dataMin != -1)
			dataMin = other.dataMin;
		if (other.dataMax != -1)
			dataMax = other.dataMax;
		if (other.amount != -1)
			amount = other.amount;
	}
	
	public short getData() {
		if (dataMin == dataMax)
			return dataMin;
		return (short) ((dataMin == -1 ? 0 : dataMin) + Skript.random.nextInt((dataMax == -1 ? 16 : dataMax) - (dataMin == -1 ? 0 : dataMin)));
	}
	
	public boolean isOfType(final ItemStack item) {
		return (typeid == -1 || item.getTypeId() == typeid) && (dataMin == -1 || item.getData().getData() >= dataMin) && (dataMax == -1 || item.getData().getData() <= dataMax)
				&& (amount == -1 || item.getAmount() >= amount);
	}
	
	/**
	 * @return "typeid:datamin-datamax*amount"
	 */
	@Override
	public String toString() {
		final String s = (typeid == -1 ? "" : "" + Material.getMaterial(typeid)) + (dataMin == -1 ? (dataMax != -1 ? ":0" : "") : ":" + dataMin) + (dataMax == -1 || dataMin == dataMax ? "" : "-")
				+ (dataMax == -1 || dataMin == dataMax ? "" : "" + dataMax) + (amount == -1 ? "" : "*" + amount);
		if (s.isEmpty())
			return "any";
		return s;
	}
	
	@Override
	public boolean equals(final Object obj) {
		if (obj == this)
			return true;
		if (!(obj instanceof ItemData))
			return false;
		final ItemData other = (ItemData) obj;
		return (typeid == -1 || other.typeid == -1 || other.typeid == typeid)
				&& (dataMin == -1 || other.dataMin == -1 || other.dataMin == dataMin)
				&& (dataMax == -1 || other.dataMax == -1 || other.dataMax == dataMax)
				&& (amount == -1 || other.amount == -1 || other.amount == amount);
	}
	
	@Override
	public int hashCode() {
		return typeid ^ dataMin << 8 ^ dataMax << 16 ^ amount << 24;
	}
	
	/**
	 * computes the intersection of two ItemDatas. The data range is the real intersection of the two data ranges, while the typeid and amount will be the second one's if set, else
	 * the first one's.
	 * 
	 * @param other
	 * @return a new ItemData which is the intersection of the given types.
	 */
	public ItemData intersection(final ItemData other) {
		return new ItemData(typeid == -1 ? other.typeid : typeid,
				(short) Math.max(dataMin, other.dataMin),
				dataMax == -1 ? other.dataMax : (other.dataMax == -1 ? dataMax : (short) Math.min(dataMax, other.dataMax)),
				amount == -1 ? other.amount : amount);
	}
	
	public ItemStack getRandom() {
		final short dmin = dataMin == -1 ? 0 : dataMin;
		final short dmax = dataMax == -1 ? 15 : dataMax;
		final short dmaxS = dataMax == -1 ? Short.MAX_VALUE : dataMax;
		return new ItemStack(typeid == -1 ? Utils.getRandom(Material.values()).getId() : typeid,
				amount == -1 ? 1 : amount,
				(short) (Skript.random.nextInt(dmaxS - dmin + 1) + dmin),
				Byte.valueOf((byte) (Skript.random.nextInt(dmax - dmin + 1) + dmin)));
	}
	
	public Iterator<ItemStack> getAll() {
		if (typeid == -1) {
			return new Iterator<ItemStack>() {
				
				private final Iterator<Material> iter = Arrays.asList(Material.values()).iterator();
				
				@Override
				public boolean hasNext() {
					return iter.hasNext();
				}
				
				@Override
				public ItemStack next() {
					return new ItemStack(iter.next(), amount == -1 ? 1 : amount);
				}
				
				@Override
				public void remove() {}
				
			};
		}
		if (dataMin == -1 && dataMax == -1)
			return new SingleItemIterator<ItemStack>(new ItemStack(typeid, amount == -1 ? 1 : amount));
		return new Iterator<ItemStack>() {
			
			private short data = (short) (dataMin == -1 ? -1 : dataMin - 1);
			
			@Override
			public boolean hasNext() {
				return data < (dataMax == -1 ? 15 : dataMax);
			}
			
			@Override
			public ItemStack next() {
				data++;
				return new ItemStack(typeid, amount == -1 ? 1 : amount, data, (byte) data);
			}
			
			@Override
			public void remove() {}
			
		};
	}
	
	@Override
	public ItemData clone() {
		try {
			return (ItemData) super.clone();
		} catch (final CloneNotSupportedException e) {
			throw new RuntimeException();
		}
	}
	
}
