/*
 * This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 2.5 Switzerland License. To view a copy of this license, visit
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ch/ or send a letter to Creative Commons, 444 Castro Street, Suite 900, Mountain View, California, 94041, USA.
 */

package ch.njol.skript.data;

import java.util.List;
import java.util.Locale;
import java.util.regex.Matcher;

import org.bukkit.Bukkit;
import org.bukkit.Location;
import org.bukkit.OfflinePlayer;
import org.bukkit.TreeType;
import org.bukkit.World;
import org.bukkit.block.Block;
import org.bukkit.block.BlockFace;
import org.bukkit.command.CommandSender;
import org.bukkit.entity.Entity;
import org.bukkit.entity.LivingEntity;
import org.bukkit.entity.Player;
import org.bukkit.event.Event;
import org.bukkit.event.player.PlayerCommandPreprocessEvent;
import org.bukkit.event.server.ServerCommandEvent;
import org.bukkit.inventory.Inventory;
import org.bukkit.inventory.ItemStack;

import ch.njol.skript.CommandEvent;
import ch.njol.skript.Skript;
import ch.njol.skript.api.InitException;
import ch.njol.skript.api.Parser;
import ch.njol.skript.api.Variable;
import ch.njol.skript.api.intern.Literal;
import ch.njol.skript.util.EntityType;
import ch.njol.skript.util.ItemData;
import ch.njol.skript.util.ItemType;
import ch.njol.skript.util.Time;
import ch.njol.skript.util.Timeperiod;
import ch.njol.skript.util.Timespan;
import ch.njol.skript.util.Utils;
import ch.njol.skript.util.VariableString;
import ch.njol.skript.util.WeatherType;
import ch.njol.skript.variables.base.SimpleDefaultVariable;

/**
 * @author Peter Gttinger
 * 
 */
public class DefaultClasses {
	
	public DefaultClasses() {}
	
	static {
		Skript.addClass("object", Object.class, null, null);
	}
	
	public static final class FloatDefaultVariable extends Literal<Float> {
		public FloatDefaultVariable() {
			super(Float.valueOf(1));
		}
	}
	
	static {
		Skript.addClass("float", Float.class, FloatDefaultVariable.class, new Parser<Float>() {
			@Override
			public Float parse(final String s) {
				try {
					if (s.endsWith("%")) {
						return Float.valueOf(Float.parseFloat(s.substring(0, s.length() - 1)) / 100);
					}
					return Float.valueOf(s);
				} catch (final NumberFormatException e) {
					return null;
				}
			}
			
			@Override
			public String toString(final Float o) {
				return o.toString();
			}
		});
	}
	
	public static final class DoubleDefaultVariable extends Literal<Double> {
		public DoubleDefaultVariable() {
			super(Double.valueOf(1));
		}
	}
	
	static {
		Skript.addClass("double", Double.class, DoubleDefaultVariable.class, new Parser<Double>() {
			@Override
			public Double parse(final String s) {
				try {
					if (s.endsWith("%")) {
						return Double.valueOf(Double.parseDouble(s.substring(0, s.length() - 1)) / 100);
					}
					return Double.valueOf(s);
				} catch (final NumberFormatException e) {
					return null;
				}
			}
			
			@Override
			public String toString(final Double o) {
				return o.toString();
			}
		}, "numbers?");
	}
	
	public static final class BooleanDefaultVariable extends Literal<Boolean> {
		public BooleanDefaultVariable() {
			super(Boolean.TRUE);
		}
	}
	
	static {
		Skript.addClass("boolean", Boolean.class, BooleanDefaultVariable.class, new Parser<Boolean>() {
			@Override
			public Boolean parse(final String s) {
				final int i = Utils.parseBooleanNoError(s);
				if (i == 1)
					return Boolean.TRUE;
				if (i == 0)
					return Boolean.FALSE;
				return null;
			}
			
			@Override
			public String toString(final Boolean o) {
				return o.toString();
			}
		});
	}
	
	public static final class ByteDefaultVariable extends Literal<Byte> {
		public ByteDefaultVariable() {
			super(Byte.valueOf((byte) 1));
		}
	}
	
	static {
		Skript.addClass("byte", Byte.class, ByteDefaultVariable.class, new Parser<Byte>() {
			@Override
			public Byte parse(final String s) {
				try {
					return Byte.valueOf(s);
				} catch (final NumberFormatException e) {
					return null;
				}
			}
			
			@Override
			public String toString(final Byte o) {
				return o.toString();
			}
		});
	}
	
	public static final class ShortDefaultVariable extends Literal<Short> {
		public ShortDefaultVariable() {
			super(Short.valueOf((short) 1));
		}
	}
	
	static {
		Skript.addClass("short", Short.class, ShortDefaultVariable.class, new Parser<Short>() {
			@Override
			public Short parse(final String s) {
				try {
					return Short.valueOf(s);
				} catch (final NumberFormatException e) {
					return null;
				}
			}
			
			@Override
			public String toString(final Short o) {
				return o.toString();
			}
		});
	}
	
	public static final class IntegerDefaultVariable extends Literal<Integer> {
		public IntegerDefaultVariable() {
			super(Integer.valueOf(1));
		}
	}
	
	static {
		Skript.addClass("integer", Integer.class, IntegerDefaultVariable.class, new Parser<Integer>() {
			@Override
			public Integer parse(final String s) {
				try {
					return Integer.valueOf(s);
				} catch (final NumberFormatException e) {
					return null;
				}
			}
			
			@Override
			public String toString(final Integer o) {
				return o.toString();
			}
		}, "integers?");
	}
	
	public static final class LongDefaultVariable extends Literal<Long> {
		public LongDefaultVariable() {
			super(Long.valueOf(1));
		}
	}
	
	static {
		Skript.addClass("long", Long.class, LongDefaultVariable.class, new Parser<Long>() {
			@Override
			public Long parse(final String s) {
				try {
					return Long.valueOf(s);
				} catch (final NumberFormatException e) {
					return null;
				}
			}
			
			@Override
			public String toString(final Long o) {
				return o.toString();
			}
		});
	}
	
	static {
		Skript.addClass("string", String.class, null, new Parser<String>() {
			@Override
			public String parse(final String s) {
				if (!s.startsWith("\"") || !s.endsWith("\""))
					return null;
				return s.substring(1, s.length() - 1);
			}
			
			@Override
			public String toString(final String s) {
				return s;
			}
		});
	}
	
	static {
		Skript.addClass("variablestring", VariableString.class, null, new Parser<VariableString>() {
			
			@Override
			public VariableString parse(final String s) {
				if (!s.startsWith("\"") || !s.endsWith("\""))
					return null;
				return new VariableString(s.substring(1, s.length() - 1));
			}
			
			@Override
			public String toString(final VariableString o) {
				return "";
			}
			
		});
	}
	
	public static final class ItemStackDefaultVariable extends SimpleDefaultVariable<ItemStack> {
		public ItemStackDefaultVariable() {
			super(ItemStack.class);
		}
	}
	
	static {
		Skript.addClass("itemstack", ItemStack.class, ItemStackDefaultVariable.class, new Parser<ItemStack>() {
			@Override
			public ItemStack parse(final String s) {
				return Utils.getItemStack(s);
			}
			
			@Override
			public String toString(final ItemStack o) {
				return o.getTypeId() + (o.getDurability() != 0 ? ":" + o.getDurability() : (o.getData() != null ? ":" + o.getData().getData() : "")) + "*" + o.getAmount();
			}
		});
	}
	
	public static final class ItemTypeDefaultVariable extends Literal<ItemType> {
		public ItemTypeDefaultVariable() {
			super(new ItemType(new ItemData()));
		}
	}
	
	static {
		Skript.addClass("itemtype", ItemType.class, ItemTypeDefaultVariable.class, new Parser<ItemType>() {
			@Override
			public ItemType parse(final String s) {
				return Utils.getItemType(s);
			}
			
			@Override
			public String toString(final ItemType o) {
				return o.toString();
			}
		}, "item ?types?", "items?");
	}
	
	public static final class TimeDefaultVariable extends Variable<Time> {
		
		@Override
		public void init(final List<Variable<?>> vars, final int matchedPattern, final Matcher matcher) throws InitException {}
		
		@Override
		public Time[] getAll(final Event e) {
			final World w = Skript.getEventValue(e, World.class);
			if (w == null)
				return null;
			return new Time[] {new Time((int) w.getTime())};
		}
		
		@Override
		public Class<Time> getReturnType() {
			return Time.class;
		}
		
		@Override
		public String getDebugMessage(final Event e) {
			return getAll(e)[0].toString();
		}
		
	}
	
	static {
		Skript.addClass("time", Time.class, TimeDefaultVariable.class, new Parser<Time>() {
			
			@Override
			public Time parse(final String s) {
				return Time.parse(s);
			}
			
			@Override
			public String toString(final Time t) {
				return t.toString();
			}
			
		}, "times?");
	}
	
	static {
		new Timespan();
	}
	
	public static final class TimeperiodDefaultVariable extends Literal<Timeperiod> {
		public TimeperiodDefaultVariable() {
			super(new Timeperiod(0, 23999));
		}
	}
	
	static {
		Skript.addClass("timeperiod", Timeperiod.class, TimeperiodDefaultVariable.class, new Parser<Timeperiod>() {
			@Override
			public Timeperiod parse(final String s) {
				if (s.equalsIgnoreCase("day")) {
					return new Timeperiod(0, 11999);
				} else if (s.equalsIgnoreCase("dusk")) {
					return new Timeperiod(12000, 13799);
				} else if (s.equalsIgnoreCase("night")) {
					return new Timeperiod(13800, 22199);
				} else if (s.equalsIgnoreCase("dawn")) {
					return new Timeperiod(22200, 23999);
				}
				final int c = s.indexOf('-');
				try {
					if (c == -1) {
						final Time t = Time.parse(s);
						if (t == null)
							return null;
						return new Timeperiod(t.getTicks());
					}
					final Time t1 = Time.parse(s.substring(0, c));
					final Time t2 = Time.parse(s.substring(c + 1));
					if (t1 == null || t2 == null)
						return null;
					return new Timeperiod(t1.getTicks(), t2.getTicks());
				} catch (final NumberFormatException e) {
					return null;
				}
			}
			
			@Override
			public String toString(final Timeperiod o) {
				return o.toString();
			}
		}, "time ?periods?", "durations?");
	}
	
	public static final class TreeTypeDefaultVariable extends Literal<TreeType> {
		public TreeTypeDefaultVariable() {
			super(TreeType.TREE);
		}
	}
	
	static {
		Skript.addClass("treetype", TreeType.class, TreeTypeDefaultVariable.class, new Parser<TreeType>() {
			
			@Override
			public TreeType parse(final String s) {
				try {
					return TreeType.valueOf(s.toUpperCase(Locale.ENGLISH).replace(' ', '_'));
				} catch (final IllegalArgumentException e) {
					return null;
				}
			}
			
			@Override
			public String toString(final TreeType o) {
				return o.toString().toLowerCase().replace('_', ' ');
			}
			
		}, "tree ?types?", "trees?");
	}
	
	public static final class WeatherTypeDefaultVariable extends Literal<WeatherType> {
		public WeatherTypeDefaultVariable() {
			super(WeatherType.clear);
		}
	}
	
	static {
		Skript.addClass("weathertype", WeatherType.class, WeatherTypeDefaultVariable.class, new Parser<WeatherType>() {
			
			@Override
			public WeatherType parse(final String s) {
				return WeatherType.parse(s);
			}
			
			@Override
			public String toString(final WeatherType o) {
				return o.toString();
			}
			
		}, "weather ?types?", "weather conditions", "weathers?");
	}
	
	public static final class EntityTypeDefaultVariable extends Literal<EntityType> {
		public EntityTypeDefaultVariable() {
			super(new EntityType(Entity.class, 1));
		}
	}
	
	static {
		Skript.addClass("entitytype", EntityType.class, EntityTypeDefaultVariable.class, new Parser<EntityType>() {
			@Override
			public EntityType parse(String s) {
				if (s.matches("\\d+ .+")) {
					if (s.endsWith("s"))
						s = s.substring(0, s.length() - 1);
					return new EntityType(Utils.getEntityType(s.split(" ", 2)[1]), Integer.parseInt(s.split(" ", 2)[0]));
				}
				return new EntityType(Utils.getEntityType(s), 1);
			}
			
			@Override
			public String toString(final EntityType o) {
				return o.toString();
			}
		}, "entity ?types?", "enit(y|ies)");
	}
	
	public static final class EntityDefaultVariable extends SimpleDefaultVariable<Entity> {
		public EntityDefaultVariable() {
			super(Entity.class);
		}
	}
	
	static {
		Skript.addClass("entity", Entity.class, EntityDefaultVariable.class, null);
	}
	
	public static final class LivingEntityDefaultVariable extends SimpleDefaultVariable<LivingEntity> {
		public LivingEntityDefaultVariable() {
			super(LivingEntity.class);
		}
	}
	
	static {
		Skript.addClass("livingentity", LivingEntity.class, LivingEntityDefaultVariable.class, null);
	}
	
	public static final class BlockDefaultVariable extends SimpleDefaultVariable<Block> {
		public BlockDefaultVariable() {
			super(Block.class);
		}
	}
	
	static {
		Skript.addClass("block", Block.class, BlockDefaultVariable.class, null);
	}
	
	public static final class LocationDefaultVariable extends SimpleDefaultVariable<Location> {
		public LocationDefaultVariable() {
			super(Location.class);
		}
	}
	
	static {
		Skript.addClass("location", Location.class, LocationDefaultVariable.class, null);
	}
	
	public static final class WorldDefaultVariable extends SimpleDefaultVariable<World> {
		public WorldDefaultVariable() {
			super(World.class);
		}
	}
	
	static {
		Skript.addClass("world", World.class, WorldDefaultVariable.class, new Parser<World>() {
			@Override
			public World parse(final String s) {
				return Bukkit.getServer().getWorld(s);
			}
			
			@Override
			public String toString(final World o) {
				return o.getName();
			}
		}, "worlds?");
	}
	
	public static final class InventoryDefaultVariable extends SimpleDefaultVariable<Inventory> {
		public InventoryDefaultVariable() {
			super(Inventory.class);
		}
	}
	
	static {
		Skript.addClass("inventory", Inventory.class, InventoryDefaultVariable.class, null);
	}
	
	public static final class PlayerDefaultVariable extends SimpleDefaultVariable<Player> {
		public PlayerDefaultVariable() {
			super(Player.class);
		}
	}
	
	static {
		Skript.addClass("player", Player.class, PlayerDefaultVariable.class, new Parser<Player>() {
			@Override
			public Player parse(final String s) {
				// if (!s.matches("\"\\S+\""))
				return null;
				// return Bukkit.getPlayerExact(s.substring(1, s.length() - 1));
			}
			
			@Override
			public String toString(final Player p) {
				return p.getDisplayName();
			}
		}, "players?");
	}
	
	public static final class OfflinePlayerDefaultVariable extends SimpleDefaultVariable<OfflinePlayer> {
		public OfflinePlayerDefaultVariable() {
			super(OfflinePlayer.class);
		}
	}
	
	static {
		Skript.addClass("offlineplayer", OfflinePlayer.class, OfflinePlayerDefaultVariable.class, new Parser<OfflinePlayer>() {
			@Override
			public OfflinePlayer parse(final String s) {
				if (!s.matches("\"\\S+\""))
					return null;
				return Bukkit.getOfflinePlayer(s.substring(1, s.length() - 1));
			}
			
			@Override
			public String toString(final OfflinePlayer p) {
				return p.getName();
			}
		}, "players?");
	}
	
	public static final class CommandSenderDefaultVariable extends Variable<CommandSender> {
		
		@Override
		public void init(final List<Variable<?>> vars, final int matchedPattern, final Matcher matcher) throws InitException {}
		
		@Override
		public CommandSender[] getAll(final Event e) {
			if (e instanceof ServerCommandEvent) {
				return new CommandSender[] {((ServerCommandEvent) e).getSender()};
			} else if (e instanceof PlayerCommandPreprocessEvent) {
				return new CommandSender[] {((PlayerCommandPreprocessEvent) e).getPlayer()};
			} else if (e instanceof CommandEvent) {
				return new CommandSender[] {((CommandEvent) e).getSender()};
			}
			return new CommandSender[] {Bukkit.getConsoleSender()};
		}
		
		@Override
		public Class<CommandSender> getReturnType() {
			return CommandSender.class;
		}
		
		@Override
		public String getDebugMessage(final Event e) {
			return getAll(e)[0].getName();
		}
		
	}
	
	static {
		Skript.addClass("commandsender", CommandSender.class, CommandSenderDefaultVariable.class, new Parser<CommandSender>() {
			@Override
			public CommandSender parse(final String s) {
				if (s.equalsIgnoreCase("console") || s.equalsIgnoreCase("server"))
					return Bukkit.getConsoleSender();
				return Bukkit.getServer().getPlayerExact(s);
			}
			
			@Override
			public String toString(final CommandSender o) {
				if (o instanceof Player)
					return ((Player) o).getDisplayName();
				return o.getName();
			}
		});
	}
	
	public static final class BlockFaceDefaultVariable extends SimpleDefaultVariable<BlockFace> {
		public BlockFaceDefaultVariable() {
			super(BlockFace.class);
		}
	}
	
	static {
		Skript.addClass("direction", BlockFace.class, BlockFaceDefaultVariable.class, new Parser<BlockFace>() {
			@Override
			public BlockFace parse(final String s) {
				return Utils.getBlockFace(s, true);
			}
			
			@Override
			public String toString(final BlockFace o) {
				return o.toString().toLowerCase(Locale.ENGLISH).replace('_', ' ');
			}
		}, "directions?");
		
	}
}
