/*
 * This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 2.5 Switzerland License. To view a copy of this license, visit
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ch/ or send a letter to Creative Commons, 444 Castro Street, Suite 900, Mountain View, California, 94041, USA.
 */

package ch.njol.skript.config;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import ch.njol.skript.Skript;
import ch.njol.skript.SkriptLogger;
import ch.njol.skript.api.intern.SkriptAPIException;

public class SectionNode extends ConfigNode implements Iterable<ConfigNode> {
	
	private final ArrayList<ConfigNode> nodes = new ArrayList<ConfigNode>();
	
	public SectionNode(final String name, final SectionNode parent) {
		super(name, parent);
	}
	
	SectionNode(final Config c) {
		super(c);
	}
	
	@Override
	public Iterator<ConfigNode> iterator() {
		return new ConfigNodeIterator(this, true);
	}
	
	public final List<ConfigNode> getNodeList() {
		return nodes;
	}
	
	/**
	 * get a subnode with the specified name
	 * 
	 * @param name
	 * @return
	 */
	public ConfigNode get(final String name) {
		for (final ConfigNode node : nodes) {
			if (node.name != null && node.name.equals(name))
				return node;
		}
		return null;
	}
	
	/**
	 * void here = void (incl. invalid)
	 * 
	 * @return
	 */
	public int getNumVoidNodes() {
		int r = 0;
		for (final ConfigNode node : nodes) {
			if (node instanceof VoidNode)
				r++;
		}
		return r;
	}
	
	static final SectionNode load(final Config c) throws IOException {
		return new SectionNode(c).load_i();
	}
	
	static final SectionNode load(final String name, final SectionNode parent) throws IOException {
		parent.config.level++;
		final SectionNode node = new SectionNode(name, parent).load_i();
		SkriptLogger.setNode(parent);
		parent.config.level--;
		return node;
	}
	
	private final SectionNode load_i() throws IOException {
		
		while (config.r.readLine() != null) {
			
			SkriptLogger.setNode(this);
			
			String line = config.r.getLine().replaceFirst("(?<!#)#(?!#).*$", "").replace("##", "#");
			
			if (!line.matches("^\\s*$") && !line.matches("^(" + config.indentation + "){" + config.level + "}\\S.*")) {
				if (line.matches("^(" + config.indentation + "){" + config.level + "}\\s.*")) {
					nodes.add(new InvalidNode(this));
					Skript.error("indentation error, expected " + config.indentationName + " x " + config.level + ", found: '"
							+ line.replaceFirst("\\S.*$", "").replace("\t", "->").replace(' ', '_') + "' (-> = tab, _ = space)");
					continue;
				} else {
					config.r.reset();
					return this;
				}
			}
			
			line = line.trim();
			
			if (line.isEmpty()) {
				nodes.add(new VoidNode(this));
				continue;
			}
			
			if (line.startsWith("!") && line.indexOf('[') != -1 && line.endsWith("]")) {
				final String option = line.substring(1, line.indexOf('['));
				final String value = line.substring(line.indexOf('[') + 1, line.length() - 1);
				if (value.isEmpty()) {
					Skript.error("parse options must not be empty");
					nodes.add(new InvalidNode(this));
					continue;
				} else if (option.equalsIgnoreCase("separator")) {
					if (config.simple) {
						Skript.warning("trigger files don't have a separator");
						continue;
					}
					config.separator = value;
				} else if (option.equalsIgnoreCase("indentation")) {
					setIndentation(value);
				} else {
					nodes.add(new InvalidNode(this));
					Skript.error("unknown parse option '" + option + "'");
					continue;
				}
				nodes.add(new ParseOptionNode(line.substring(0, line.indexOf('[')), this));
				continue;
			}
			
			if (line.endsWith(":") && (
					config.simple
							|| line.indexOf(config.separator) == -1
							|| config.separator.endsWith(":") && line.indexOf(config.separator) == line.length() - config.separator.length()
					)) {
				nodes.add(SectionNode.load(line.substring(0, line.length() - 1), this));
				continue;
			}
			
			if (config.simple) {
				nodes.add(new SimpleNode(line, this));
			} else {
				nodes.add(getEntry(line, orig, config.r.getLineNum()));
			}
			
		}
		
		return this;
	}
	
	private final ConfigNode getEntry(final String line, final String orig, final int lineNum) {
		final int x = line.indexOf(config.separator);
		if (x == -1) {
			final InvalidNode n = new InvalidNode(this, line, lineNum);
			Skript.error("missing separator '" + config.separator + "'");
			SkriptLogger.setNode(this);
			return n;
		}
		final String key = line.substring(0, x).trim();
		final String value = line.substring(x + config.separator.length()).trim();
		return new EntryNode(key, value, orig, this, config.r.getLineNum());
	}
	
	public void convertToEntries(final int levels) {
		if (!config.simple)
			throw new SkriptAPIException("config is not simple");
		for (int i = 0; i < nodes.size(); i++) {
			final ConfigNode n = nodes.get(i);
			if (levels > 0 && n.isSection()) {
				((SectionNode) n).convertToEntries(levels - 1);
			}
			if (!(n instanceof SimpleNode))
				continue;
			nodes.set(i, getEntry(n.getName(), n.getOrig(), n.lineNum));
		}
	}
	
	@Override
	void save(final PrintWriter w) {
		if (parent != null) {
			if (!modified) {
				w.println(getIndentation() + orig.trim());
			} else {
				w.println(getIndentation() + name + ":" + getComment());
			}
		}
		for (final ConfigNode node : nodes)
			node.save(w);
		modified = false;
	}
	
}
