/*
 * This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 2.5 Switzerland License. To view a copy of this license, visit
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ch/ or send a letter to Creative Commons, 444 Castro Street, Suite 900, Mountain View, California, 94041, USA.
 */

package ch.njol.skript.api;

import java.lang.reflect.Array;
import java.util.Iterator;

import org.bukkit.event.Event;

import ch.njol.skript.Skript;
import ch.njol.skript.api.Changer.ChangeMode;
import ch.njol.skript.api.intern.ConvertedVariable;
import ch.njol.skript.api.intern.Literal;
import ch.njol.skript.util.Utils;
import ch.njol.util.Checker;

/**
 * 
 * A variable.
 * 
 * @author Peter Gttinger
 * 
 */
public abstract class Variable<T> implements Expression, Debuggable {
	
	protected boolean and;
	
	protected Variable() {}
	
	/**
	 * DO NOT USE THIS IN CONDITIONS, use check() instead.
	 * 
	 * @param e
	 * @return may return null
	 */
	@SuppressWarnings("unchecked")
	public final T[] get(final Event e) {
		final T[] all = getAll(e);
		if (all == null)
			return (T[]) Array.newInstance(this.getReturnType(), 1);
		if (and || all.length <= 1)
			return all;
		final T r = Utils.getRandom(all);
		final T[] one = (T[]) Array.newInstance(r.getClass(), 1);
		one[0] = r;
		return one;
	}
	
	/**
	 * 
	 * @param e
	 * @return An array holding all values. Must not be null.
	 */
	protected abstract T[] getAll(Event e);
	
	/**
	 * Checks through the values to find you whether this variable matches the checker.
	 * 
	 * The argument of the checker is guaranteed to never be null.
	 * 
	 * @param e
	 * @param c
	 * @return
	 */
	public final boolean check(final Event e, final Checker<T> c, final Condition cond) {
		return check(e, c, cond.isNegated(), false);
	}

	public final boolean check(final Event e, final Checker<T> c, final boolean invert, boolean includeNull) {
		for (final T o : getAll(e)) {
			final boolean b = includeNull ? c.check(o) : (o == null ? false : c.check(o));
			if (invert) {
				if (and && b)
					return false;
				if (!and && !b)
					return true;
			} else {
				if (and && !b)
					return false;
				if (!and && b)
					return true;
			}
		}
		return and;
	}

	/**
	 * 
	 * @param to
	 * @return variable with the desired return type or null if no converter exists
	 */
	@SuppressWarnings("unchecked")
	public <R> Variable<R> getConvertedVar(final Class<R> to) {
		if (getReturnType().isAssignableFrom(to))
			return (Variable<R>) this;
		return ConvertedVariable.newInstance(this, to);
	}
	
	public static final <T> Variable<? extends T> parse(final String s, final Class<T> returnType) {
		return parse(s, returnType, Skript.getVariables().listIterator());
	}
	
	public static final Variable<?> parseNoLiteral(final String s, final Iterator<? extends VariableInfo<?>> source) {
		final Variable<?> var = (Variable<?>) Expressions.parse(s, source);
		
		if (var != null) {
			Skript.clearErrorCause();
			return var;
		}
		
		return null;
	}
	
	@SuppressWarnings("unchecked")
	public static final <T> Variable<? extends T> parse(final String s, final Class<T> returnType, final Iterator<? extends VariableInfo<?>> source) {
		
		final Variable<?> v = parseNoLiteral(s, source);
		if (v != null) {
			if (returnType.isAssignableFrom(v.getReturnType()))
				return (Variable<? extends T>) v;
			return v.getConvertedVar(returnType);
		}
		
		if (returnType == Object.class)
			return (Variable<? extends T>) Expressions.parseLiteralArray(s);
		final Literal<T> t = Expressions.parseLiteralArray(s).getConvertedVar(returnType);
		if (t != null) {
			Skript.clearErrorCause();
			return t;
		}
		
		Skript.setErrorCause("'" + s + "' is invalid or cannot be used here", true);
		return null;
	}
	
	/**
	 * this is set automatically and should not be changed.
	 * 
	 * @param and
	 */
	public void setAnd(final boolean and) {
		this.and = and;
	}
	
	// TODO
//	@Override
//	public abstract String toString();
	
	public abstract Class<? extends T> getReturnType();
	
	/**
	 * Changes the variable's value by the given amount. This will only be called on supported modes and with the desired <code>delta</code> type as returned by {@link #acceptChange(ChangeMode)}<br/>
	 * The default implementation of this method throws an exception at runtime.
	 * 
	 * @param e
	 * @param delta the amount to vary this variable by
	 * @param mode
	 */
	public void change(final Event e, final Object[] delta, final ch.njol.skript.api.Changer.ChangeMode mode) {
		throw new UnsupportedOperationException();
	}
	
	/**
	 * tests whether this variable supports the given mode, and if yes what type it expects the <code>delta</code> to be.<br/>
	 * The default implementation returns null, i.e. it rejects any change attempts to this variable.
	 * @param mode
	 * @return the type that {@link #change(Event, Object[], ch.njol.skript.api.Changer.ChangeMode)} accepts as it's <code>delta</code> parameter,
	 * or null if the given mode is not supported. For {@link ChangeMode#CLEAR} this can return any class object to mark clear as supported.
	 */
	public Class<?> acceptChange(final ch.njol.skript.api.Changer.ChangeMode mode) {
		return null;
	}
	
}
