/*
 * This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 2.5 Switzerland License. To view a copy of this license, visit
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ch/ or send a letter to Creative Commons, 444 Castro Street, Suite 900, Mountain View, California, 94041, USA.
 */

package ch.njol.skript.api;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import ch.njol.skript.Skript;
import ch.njol.skript.api.SkriptEvent.SkriptEventInfo;
import ch.njol.skript.api.intern.ParsedLiteral;
import ch.njol.skript.api.intern.SkriptAPIException;
import ch.njol.util.Callback;
import ch.njol.util.RegexUtils;

/**
 * @author Peter Gttinger
 * 
 */
public interface Expression {
	
	public static abstract class ExpressionInfo {
		
		public final Class<? extends Expression> c;
		
		public final Pattern[] patterns;
		public final String[][] groups;
		
		public ExpressionInfo(final String[] patterns, final Class<? extends Expression> c) {
			this.patterns = new Pattern[patterns.length];
			groups = new String[patterns.length][];
			for (int i = 0; i < patterns.length; i++) {
				final ArrayList<String> groups = new ArrayList<String>();
				this.patterns[i] = Pattern.compile("^" + RegexUtils.replaceAll(patterns[i], "%(.+?)%", new Callback<String, Matcher>() {
					private int num = 0;
					
					@Override
					public String run(final Matcher m) {
						num++;
						groups.add(m.group(1) + num);
						return "(?<" + m.group(1) + num + ">.+?)";
					}
				}) + "$");
				this.groups[i] = groups.toArray(new String[0]);
			}
			this.c = c;
		}
	}
	
	public static class VariableInfo<T> extends ExpressionInfo {
		
		public Class<T> returnType;
		
		public VariableInfo(final String[] patterns, final Class<T> returnType, final Class<? extends Expression> c) {
			super(patterns, c);
			this.returnType = returnType;
		}
	}
	
	public static final class Expressions {
		
		public static final Expression parse(final String s, final Iterator<? extends ExpressionInfo> source) {
			Skript.clearErrorCause();
			while (source.hasNext()) {
				final Expression e = parse_i(s, source.next());
				if (e != null)
					return e;
			}
			// no error message here!
			return null;
		}
		
		private final static Expression parse_i(final String s, final ExpressionInfo info) {
			for (int i = 0; i < info.patterns.length; i++) {
				final Matcher m = info.patterns[i].matcher(s);
				if (m.matches()) {
					final ArrayList<Variable<?>> parts = new ArrayList<Variable<?>>(info.groups.length);
					for (final String t : info.groups[i]) {
						final String g = m.group(t);
						if (g == null) {
							try {
								final Class<? extends Variable<?>> dv = Skript.getDefaultVariable(t.replaceFirst("\\d+$", ""));
								parts.add(dv == null ? null : dv.newInstance());
							} catch (final InstantiationException e) {
								SkriptAPIException.instantiationException("the default variable", info.c, e);
							} catch (final IllegalAccessException e) {
								SkriptAPIException.inaccessibleConstructor(info.c, e);
							}
						} else {
							final Variable<?> p = Variable.parse(g, Skript.getClass(t.replaceFirst("\\d+$", "")));
							if (p == null)
								return null;
							parts.add(p);
						}
					}
					try {
						final Expression p = info.c.newInstance();
						try {
							p.init(parts, i, m);
						} catch (final InitException e) {
							continue;
						} catch (final ParseException e) {
							return null;
						}
						Skript.clearErrorCause();
						return p;
					} catch (final InstantiationException e) {
						SkriptAPIException.instantiationException("the variable", info.c, e);
					} catch (final IllegalAccessException e) {
						SkriptAPIException.inaccessibleConstructor(info.c, e);
					}
				}
			}
			return null;
		}
		
		static final SkriptEvent parseEvent(String s) {
			s = s.replaceFirst("^on ", "");
			for (final SkriptEventInfo info : Skript.getEvents()) {
				for (int i = 0; i < info.patterns.length; i++) {
					final Matcher m = info.patterns[i].matcher(s);
					if (m.matches()) {
						final Object[][] parts = new Object[info.groups[i].length][];
						int j = -1;
						for (final String t : info.groups[i]) {
							j++;
							final String g = m.group(t);
							if (g == null) {
								parts[j] = null;
							} else {
								final Object[] p = parseDataArray(g, Skript.getClass(t.replaceFirst("\\d+$", "")));
								if (p == null)
									return null;
								parts[j] = p;
							}
						}
						try {
							final SkriptEvent e = info.c.newInstance();
							e.init(parts, i, m);
							Skript.clearErrorCause();
							return e;
						} catch (final InstantiationException e) {
							SkriptAPIException.instantiationException("the skript event", info.c, e);
						} catch (final IllegalAccessException e) {
							SkriptAPIException.inaccessibleConstructor(info.c, e);
						}
					}
				}
			}
			return null;
		}
		
		protected final static ParsedLiteral parseLiteralArray(final String s) {
			final boolean and = !s.matches(" or ");
			final String[] p = s.split(and ? ",|,? and " : ",|,? or ");
			return new ParsedLiteral(p, and);
		}
		
		@SuppressWarnings("unchecked")
		public final static <T> T[] parseDataArray(final String s, final Class<T> c) {
			if (c == Object.class)
				throw new SkriptAPIException("parseDataArray(String, Class) must be called with the exact class to parse");
			final boolean and = !s.matches(" or ");
			final String[] p = s.split(and ? ",|,? and " : ",|,? or ");
			final T[] data = (T[]) Array.newInstance(c, p.length);
			final Converter<String, ? extends T> parser = Skript.getParser(c);
			if (parser == null)
				throw new SkriptAPIException("no parser registered for " + c.getName());
			for (int i = 0; i < p.length; i++) {
				final T t = parser.convert(p[i].trim());
				if (t == null) {
					Skript.setErrorCause("'" + p[i] + "' is invalid or cannot be used here", true);
					return null;
				}
				data[i] = t;
			}
			Skript.clearErrorCause();
			
			return data;
		}
		
	}
	
	/**
	 * called just after the constructor.
	 * 
	 * @param vars all %var%s included in the matching pattern in the order they appear in the pattern. If an optional value was left out it will still be included in this list
	 *            holding the default value of the desired type which usually depends on the event.
	 * @param matchedPattern the pattern which matched
	 * @param matcher all info about the match. Note that any (Nth) %variable% is converted to (?&lt;variableN&gt;.+?) and is thus a group which has an index.
	 * 
	 * @throws InitException throwing this has the same effect as if no pattern matched. This is an exception, meaning it should only be thrown in exceptional cases where a regex
	 *             is not enough.
	 * @throws ParseException throw this if some part of the expression was parsed and found to be invalid, but the whole expression still matched correctly.<br/>
	 *             This will immediately print an error and it's cause which is set to the cause passed to the exception.
	 */
	public void init(List<Variable<?>> vars, int matchedPattern, Matcher matcher) throws InitException, ParseException;
	
}
