/*
 * This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 2.5 Switzerland License. To view a copy of this license, visit
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ch/ or send a letter to Creative Commons, 444 Castro Street, Suite 900, Mountain View, California, 94041, USA.
 */

package ch.njol.skript;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.bukkit.event.Event;

import ch.njol.skript.api.Condition;
import ch.njol.skript.api.LoopVar;
import ch.njol.skript.api.SkriptEvent;
import ch.njol.skript.api.Variable;
import ch.njol.skript.api.intern.Conditional;
import ch.njol.skript.api.intern.Loop;
import ch.njol.skript.api.intern.TopLevelExpression;
import ch.njol.skript.api.intern.TriggerItem;
import ch.njol.skript.api.intern.TriggerSection;
import ch.njol.skript.config.Config;
import ch.njol.skript.config.ConfigNode;
import ch.njol.skript.config.EntryNode;
import ch.njol.skript.config.SectionNode;
import ch.njol.skript.config.SimpleNode;
import ch.njol.util.Pair;

/**
 * 
 * @author Peter Gttinger
 * 
 */
final public class TriggerFileLoader {
	private TriggerFileLoader() {}
	
	public static SkriptEvent currentEvent;
	public static ArrayList<TriggerSection> currentSections = new ArrayList<TriggerSection>();
	public static ArrayList<Pair<String, LoopVar<?>>> currentLoops = new ArrayList<Pair<String, LoopVar<?>>>();
	
	private static String indentation = "";
	
	@SuppressWarnings("unchecked")
	public static ArrayList<TriggerItem> loadItems(final SectionNode node) {
		if (Skript.logExtreme())
			indentation += "    ";
		final ArrayList<TriggerItem> items = new ArrayList<TriggerItem>();
		for (final ConfigNode n : node) {
			if (n.isVoid())
				continue;
			SkriptLogger.setNode(n);
			if (n.isEntry()) {
				final SimpleNode e = (SimpleNode) n;
				
				final TopLevelExpression expr = TopLevelExpression.parse(e.getName());
				if (expr == null) {
					Skript.printErrorAndCause("can't understand this condition/effect: '" + e.getName() + "'");
					continue;
				}
				if (Skript.logExtreme())
					Skript.info(indentation + expr.getDebugMessage(null));
				items.add(expr);
			} else if (n.isSection()) {
				if (n.getName().startsWith("loop ")) {
					final LoopVar<?> loopvar = (LoopVar<?>) Variable.parseNoLiteral(n.getName().substring("loop ".length()), Skript.loops.listIterator());
					if (loopvar == null) {
						Skript.printErrorAndCause("can't understand this loop: '" + n.getName() + "'");
						continue;
					}
					if (Skript.logExtreme())
						Skript.info(indentation + "loop " + loopvar.getDebugMessage(null) + ":");
					currentLoops.add(new Pair<String, LoopVar<?>>(Skript.getLoopVarName((Class<? extends LoopVar<?>>) loopvar.getClass()), loopvar));
					final Loop loop = new Loop(loopvar, (SectionNode) n);
					currentLoops.remove(currentLoops.size() - 1);
					items.add(loop);
				} else if (n.getName().equalsIgnoreCase("else")) {
					Skript.warning("else is not yet implemented");
				} else {
					final Condition cond = Condition.parse(n.getName());
					if (cond == null) {
						Skript.printErrorAndCause("can't understand this condition: '" + n.getName() + "'");
						continue;
					}
					if (Skript.logExtreme())
						Skript.info(indentation + cond.getDebugMessage(null) + ":");
					final Conditional c = new Conditional(cond, (SectionNode) n);
					items.add(c);
				}
			}
		}
		SkriptLogger.setNode(node);
		if (Skript.logExtreme())
			indentation = indentation.substring(0, indentation.length() - 4);
		return items;
	}
	
	static int load(final Config config) {
		int i = 0;
		for (final ConfigNode cnode : config.getMainNode()) {
			if (cnode.isVoid())
				continue;
			if (!cnode.isSection()) {
				Skript.error("invalid line");
				continue;
			}
			final SectionNode node = ((SectionNode) cnode);
			if (node.getName().startsWith("command")) {
				if (Skript.commandMap == null) {
					Skript.error("CraftBukkit is required to create new commands");
					continue;
				}
				final Matcher m = Pattern.compile("command\\s+(\\S+)").matcher(node.getName());
				if (!m.find()) {
					Skript.error("invalid command layout, use 'command cmd:' where cmd is your command (which must not contain spaces)");
					continue;
				}
				String command = m.group(1);
				if (command.startsWith("/"))
					command = command.substring(1);
				node.convertToEntries(0);
				String description = "", usage = "/" + command, permission = null, permissionMessage = null;
				List<String> aliases = null;
				SectionNode trigger = null;
				for (final ConfigNode n : node) {
					if (n.getName().equalsIgnoreCase("usage")) {
						if (n.isEntry())
							usage = ((EntryNode) n).getValue();
					} else if (n.getName().equalsIgnoreCase("description")) {
						if (n.isEntry())
							description = ((EntryNode) n).getValue();
					} else if (n.getName().equalsIgnoreCase("aliases")) {
						if (n.isEntry())
							aliases = Arrays.asList(((EntryNode) n).getValue().split("\\s*,\\s*"));
					} else if (n.getName().equalsIgnoreCase("permission")) {
						if (n.isEntry())
							permission = ((EntryNode) n).getValue();
					} else if (n.getName().equalsIgnoreCase("permission message")) {
						if (n.isEntry())
							permissionMessage = ((EntryNode) n).getValue();
					} else if (n.getName().equalsIgnoreCase("trigger")) {
						if (n.isSection())
							trigger = (SectionNode) n;
					} else {
						Skript.error("invalid entry '" + n.getName() + "' in command");
					}
				}
				if (trigger == null) {
					Skript.error("A command must always have a 'trigger:' section where all effects & conditions are put");
					continue;
				}
				if (permissionMessage != null && permission == null) {
					Skript.warning("command /" + command + " has a permission message set, but not a permission.");
				}
				if (Skript.logExtreme())
					Skript.info("command /" + command + ":");
				Skript.addCommand(new SkriptCommand(command, description, usage, aliases, permission, permissionMessage, loadItems(trigger)));
				if (Skript.logVeryHigh() && !Skript.logExtreme())
					Skript.info("registered command /" + command);
				continue;
			}
			if (Skript.logVeryHigh() && !Skript.logExtreme())
				Skript.info("loading trigger '" + node.getName() + "'");
			final SkriptEvent event = SkriptEvent.parse(node.getName());
			if (event == null) {
				Skript.printErrorAndCause("can't understand this event: '" + node.getName() + "'");
				continue;
			}
			else if (Skript.logExtreme())
				Skript.info(node.getName() + " (" + event.getDebugMessage(null) + "):");
			currentEvent = event;
			final Class<? extends Event>[] events = Skript.getEvents(event.getClass());
			final Trigger trigger = new Trigger(node.getName(), event, loadItems(node));
			EventHandler.addTrigger(events, trigger);
			i++;
		}
		if (Skript.logHigh())
			Skript.info("loaded " + i + " trigger" + (i == 1 ? "" : "s") + (Skript.logVeryHigh() ? "" : " from '" + config.getFileName() + "'"));
		return i;
	}
	
	public static String[] split(final String s) {
		int c = -1, p = -1;
		final ArrayList<String> r = new ArrayList<String>();
		while (c != s.length()) {
			final int oldc = c;
			p = s.indexOf('%', c + 1);
			if (p == -1)
				p = s.length();
			c = s.indexOf(',', c + 1);
			if (c == -1)
				c = s.length();
			if (p < c) {
				p = s.indexOf('%', p + 1);
				if (c < p) {
					c = s.indexOf(',', p + 1);
					if (c == -1)
						c = s.length();
				}
			}
			r.add(s.substring(oldc + 1, c).trim());
		}
		return r.toArray(new String[r.size()]);
	}
	
}
