/*
 * This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 2.5 Switzerland License. To view a copy of this license, visit
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ch/ or send a letter to Creative Commons, 444 Castro Street, Suite 900, Mountain View, California, 94041, USA.
 */

package ch.njol.skript.util;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.bukkit.Bukkit;
import org.bukkit.Location;
import org.bukkit.Material;
import org.bukkit.World;
import org.bukkit.block.Block;
import org.bukkit.block.BlockFace;
import org.bukkit.block.BlockState;
import org.bukkit.entity.Animals;
import org.bukkit.entity.Arrow;
import org.bukkit.entity.Boat;
import org.bukkit.entity.CaveSpider;
import org.bukkit.entity.Chicken;
import org.bukkit.entity.Cow;
import org.bukkit.entity.Creature;
import org.bukkit.entity.Creeper;
import org.bukkit.entity.Egg;
import org.bukkit.entity.Enderman;
import org.bukkit.entity.Entity;
import org.bukkit.entity.ExperienceOrb;
import org.bukkit.entity.Explosive;
import org.bukkit.entity.FallingSand;
import org.bukkit.entity.Fireball;
import org.bukkit.entity.Fish;
import org.bukkit.entity.Flying;
import org.bukkit.entity.Ghast;
import org.bukkit.entity.Giant;
import org.bukkit.entity.HumanEntity;
import org.bukkit.entity.Item;
import org.bukkit.entity.LightningStrike;
import org.bukkit.entity.LivingEntity;
import org.bukkit.entity.Minecart;
import org.bukkit.entity.Monster;
import org.bukkit.entity.Painting;
import org.bukkit.entity.Pig;
import org.bukkit.entity.PigZombie;
import org.bukkit.entity.Player;
import org.bukkit.entity.PoweredMinecart;
import org.bukkit.entity.Projectile;
import org.bukkit.entity.Sheep;
import org.bukkit.entity.Silverfish;
import org.bukkit.entity.Skeleton;
import org.bukkit.entity.Slime;
import org.bukkit.entity.Snowball;
import org.bukkit.entity.Spider;
import org.bukkit.entity.Squid;
import org.bukkit.entity.StorageMinecart;
import org.bukkit.entity.TNTPrimed;
import org.bukkit.entity.Vehicle;
import org.bukkit.entity.WaterMob;
import org.bukkit.entity.Wolf;
import org.bukkit.entity.Zombie;
import org.bukkit.event.Event;
import org.bukkit.event.block.BlockBreakEvent;
import org.bukkit.event.block.BlockDamageEvent;
import org.bukkit.event.block.BlockEvent;
import org.bukkit.event.block.BlockPlaceEvent;
import org.bukkit.event.block.SignChangeEvent;
import org.bukkit.event.entity.EntityDamageByBlockEvent;
import org.bukkit.event.entity.EntityDamageByEntityEvent;
import org.bukkit.event.entity.EntityEvent;
import org.bukkit.event.entity.EntityRegainHealthEvent;
import org.bukkit.event.entity.EntityTargetEvent;
import org.bukkit.event.entity.ProjectileHitEvent;
import org.bukkit.event.inventory.FurnaceBurnEvent;
import org.bukkit.event.inventory.FurnaceSmeltEvent;
import org.bukkit.event.painting.PaintingEvent;
import org.bukkit.event.painting.PaintingPlaceEvent;
import org.bukkit.event.player.PlayerBedEnterEvent;
import org.bukkit.event.player.PlayerBedLeaveEvent;
import org.bukkit.event.player.PlayerBucketEvent;
import org.bukkit.event.player.PlayerEvent;
import org.bukkit.event.player.PlayerInteractEntityEvent;
import org.bukkit.event.player.PlayerInteractEvent;
import org.bukkit.event.vehicle.VehicleDamageEvent;
import org.bukkit.event.vehicle.VehicleDestroyEvent;
import org.bukkit.event.vehicle.VehicleEnterEvent;
import org.bukkit.event.vehicle.VehicleEvent;
import org.bukkit.event.vehicle.VehicleExitEvent;
import org.bukkit.event.weather.WeatherEvent;
import org.bukkit.event.world.WorldEvent;
import org.bukkit.inventory.Inventory;
import org.bukkit.inventory.ItemStack;
import org.bukkit.util.Vector;

import ch.njol.skript.Skript;
import ch.njol.skript.SkriptLogger;
import ch.njol.skript.api.LoopEvent;
import ch.njol.skript.events.ScheduledEvent;
import ch.njol.util.Callback;

/**
 * 
 * Utility class. these functions are really useful if you intend to make new conditions, effects and variables.
 * 
 * @author Peter Gttinger
 * 
 */
public abstract class Utils {
	private Utils() {}
	
	public static boolean parseBoolean(final String s) {
		boolean r = (s.equalsIgnoreCase("true") || s.equalsIgnoreCase("yes"));
		if (r)
			return true;
		r = s.equalsIgnoreCase("false") || s.equalsIgnoreCase("no");
		if (r)
			return false;
		SkriptLogger.expectationError("boolean (true/yes or false/no)", s);
		return false;
	}
	
	public static int parseBooleanNoError(final String s) {
		if (s.equalsIgnoreCase("true") || s.equalsIgnoreCase("yes"))
			return 1;
		if (s.equalsIgnoreCase("false") || s.equalsIgnoreCase("no"))
			return 0;
		return -1;
	}
	
	public static short parseShort(final String s) {
		try {
			return Short.parseShort(s);
		} catch (final NumberFormatException e) {
			SkriptLogger.expectationError("integer", s);
		}
		return 0;
	}
	
	public static int parseInt(final String s) {
		try {
			return Integer.parseInt(s);
		} catch (final NumberFormatException e) {
			SkriptLogger.expectationError("integer", s);
		}
		return 0;
	}
	
	public static float parseFloat(final String s) {
		try {
			return Float.parseFloat(s);
		} catch (final NumberFormatException e) {
			SkriptLogger.expectationError("number", s);
		}
		return 0;
	}
	
	public static double parseDouble(final String s) {
		try {
			return Double.parseDouble(s);
		} catch (final NumberFormatException e) {
			SkriptLogger.expectationError("number", s);
		}
		return 0;
	}
	
	public static <T> int contains(final T[] array, final T o) {
		if (array == null)
			return -1;
		int i = 0;
		for (final T a : array) {
			if (a.equals(o))
				return i;
			i++;
		}
		return -1;
	}
	
	public static int containsIgnoreCase(final String[] array, final String s) {
		if (array == null)
			return -1;
		int i = 0;
		for (final String a : array) {
			if (a.equalsIgnoreCase(s))
				return i;
			i++;
		}
		return -1;
	}
	
	/**
	 * 
	 * Finds an object in an array using {@link Object#equals(Object)} to test equality.
	 * 
	 * @param array
	 * @param o
	 * @return
	 */
	public static <T> int contains(final Iterable<T> array, final T o) {
		if (array == null)
			return -1;
		int i = 0;
		for (final T a : array) {
			if (a.equals(o))
				return i;
			i++;
		}
		return -1;
	}
	
	/**
	 * finds a String in an array of Strings (ignoring case).
	 * 
	 * @param array
	 * @param s
	 * @return
	 */
	public static int containsIgnoreCase(final Iterable<String> array, final String s) {
		if (array == null)
			return -1;
		int i = 0;
		for (final String a : array) {
			if (a.equalsIgnoreCase(s))
				return i;
			i++;
		}
		return -1;
	}
	
	public static <T, U> Entry<T, U> containsKey(final Map<T, U> m, final Object key) {
		if (m == null)
			return null;
		for (final Entry<T, U> e : m.entrySet()) {
			if (e.getKey() == key)
				return e;
		}
		return null;
	}
	
	public static <T, U> Entry<T, U> containsEqualKey(final Map<T, U> m, final Object key) {
		if (m == null)
			return null;
		for (final Entry<T, U> e : m.entrySet()) {
			if (e.getKey().equals(key))
				return e;
		}
		return null;
	}
	
	public static <U> Entry<String, U> containsKeyIgnoreCase(final Map<String, U> m, final String key) {
		if (m == null)
			return null;
		for (final Entry<String, U> e : m.entrySet()) {
			if (e.getKey().equalsIgnoreCase(key))
				return e;
		}
		return null;
	}
	
	public static BlockFace getBlockFace(String s, final boolean printError) {
		s = s.toUpperCase(Locale.ENGLISH).replace(' ', '_');
		try {
			return BlockFace.valueOf(s);
		} catch (final IllegalArgumentException e1) {
			if (s.equals("U"))
				return BlockFace.UP;
			if (s.equals("D"))
				return BlockFace.DOWN;
			if (s.length() <= 3) {
				try {
					String r = "";
					for (int i = 0; i < s.length(); i++) {
						switch (s.charAt(i)) {
							case 'n':
								r += "NORTH_";
							break;
							case 'e':
								r += "EAST_";
							break;
							case 's':
								r += "SOUTH_";
							break;
							case 'w':
								r += "WEST_";
							break;
							default:
								if (printError)
									Skript.error("invalid direction '" + s + "'");
								return null;
						}
					}
					return BlockFace.valueOf(r.substring(0, r.length() - 1));
				} catch (final IllegalArgumentException e2) {}
			}
		}
		if (printError)
			Skript.error("invalid direction '" + s + "'");
		return null;
	}
	
	public final static String getDirectionName(final int[] dir) {
		String r = "";
		final BlockFace[] dirs = {BlockFace.NORTH, BlockFace.UP, BlockFace.EAST};
		for (final int i : new int[] {0, 2, 1}) {
			if (dir[i] == 0)
				continue;
			if (dir[i] * getBlockFaceDir(dirs[i], i) > 0)
				r += dir[i] + " " + dirs[i].toString().toLowerCase(Locale.ENGLISH);
			else
				r += dir[i] + " " + dirs[i].getOppositeFace().toString().toLowerCase(Locale.ENGLISH);
		}
		if (r.isEmpty())
			return "none";
		return r;
	}

	public static final int[] getBlockFaceDir(final BlockFace f) {
		return new int[] {f.getModX(), f.getModY(), f.getModZ()};
	}
	public static final int getBlockFaceDir(final BlockFace f, int axis) {
		return getBlockFaceDir(f)[axis];
	}
	
	public static World getWorld(final Event e) {
		if (e instanceof BlockEvent) {
			return ((BlockEvent) e).getBlock().getWorld();
		} else if (e instanceof EntityEvent) {
			return ((EntityEvent) e).getEntity().getWorld();
		} else if (e instanceof FurnaceBurnEvent) {
			return ((FurnaceBurnEvent) e).getFurnace().getWorld();
		} else if (e instanceof FurnaceSmeltEvent) {
			return ((FurnaceSmeltEvent) e).getFurnace().getWorld();
		} else if (e instanceof PaintingEvent) {
			return ((PaintingEvent) e).getPainting().getWorld();
		} else if (e instanceof PlayerEvent) {
			return ((PlayerEvent) e).getPlayer().getWorld();
		} else if (e instanceof VehicleEvent) {
			return ((VehicleEvent) e).getVehicle().getWorld();
		} else if (e instanceof WeatherEvent) {
			return ((WeatherEvent) e).getWorld();
		} else if (e instanceof WorldEvent) {
			return ((WorldEvent) e).getWorld();
		} else if (e instanceof ScheduledEvent) {
			return ((ScheduledEvent) e).getWorld();
		} else if (e instanceof LoopEvent && ((LoopEvent) e).getCurrent() instanceof World) {
			return (World) ((LoopEvent) e).getCurrent();
		}
		return null;
	}
	
	public static Block getBlock(final Event e) {
		if (e instanceof BlockEvent) {
			return ((BlockEvent) e).getBlock();
		} else if (e instanceof EntityEvent) {
			return ((EntityEvent) e).getEntity().getLocation().getBlock();
		} else if (e instanceof FurnaceBurnEvent) {
			return ((FurnaceBurnEvent) e).getFurnace();
		} else if (e instanceof FurnaceSmeltEvent) {
			return ((FurnaceSmeltEvent) e).getFurnace();
		} else if (e instanceof PaintingEvent) {
			return ((PaintingEvent) e).getPainting().getLocation().getBlock();
		} else if (e instanceof PlayerInteractEvent) {
			return ((PlayerInteractEvent) e).getClickedBlock();
		} else if (e instanceof PlayerBucketEvent) {
			return ((PlayerBucketEvent) e).getBlockClicked().getRelative(((PlayerBucketEvent) e).getBlockFace());
		} else if (e instanceof PlayerInteractEntityEvent) {
			return ((PlayerInteractEntityEvent) e).getRightClicked().getLocation().getBlock();
		} else if (e instanceof PlayerBedEnterEvent) {
			return ((PlayerBedEnterEvent) e).getBed();
		} else if (e instanceof PlayerBedLeaveEvent) {
			return ((PlayerBedLeaveEvent) e).getBed();
		} else if (e instanceof PlayerEvent) {
			return ((PlayerEvent) e).getPlayer().getLocation().getBlock();
		} else if (e instanceof VehicleEvent) {
			return ((VehicleEvent) e).getVehicle().getLocation().getBlock();
		} else if (e instanceof LoopEvent && ((LoopEvent) e).getCurrent() instanceof Block) {
			return (Block) ((LoopEvent) e).getCurrent();
		}
		return null;
	}
	
	public static Location getLocation(final Event e) {
		if (e instanceof EntityEvent) {
			return ((EntityEvent) e).getEntity().getLocation();
		} else if (e instanceof PaintingEvent) {
			return ((PaintingEvent) e).getPainting().getLocation();
		} else if (e instanceof PlayerInteractEntityEvent) {
			return ((PlayerInteractEntityEvent) e).getRightClicked().getLocation();
		} else if (e instanceof PlayerEvent) {
			return ((PlayerEvent) e).getPlayer().getLocation();
		} else if (e instanceof VehicleEvent) {
			return ((VehicleEvent) e).getVehicle().getLocation();
		} else if (e instanceof LoopEvent && ((LoopEvent) e).getCurrent() instanceof Location) {
			return (Location) ((LoopEvent) e).getCurrent();
		}
		final Block b = getBlock(e);
		if (b != null)
			return b.getLocation().add(0.5, 0.5, 0.5);
		return null;
	}
	
	public static Player getPlayer(final Event e) {
		if (e instanceof BlockBreakEvent) {
			return ((BlockBreakEvent) e).getPlayer();
		} else if (e instanceof BlockDamageEvent) {
			return ((BlockDamageEvent) e).getPlayer();
		} else if (e instanceof BlockPlaceEvent) {
			return ((BlockPlaceEvent) e).getPlayer();
		} else if (e instanceof SignChangeEvent) {
			return ((SignChangeEvent) e).getPlayer();
		} else if (e instanceof EntityDamageByEntityEvent) {
			if (((EntityDamageByEntityEvent) e).getDamager() instanceof Player)
				return (Player) ((EntityDamageByEntityEvent) e).getDamager();
		} else if (e instanceof ProjectileHitEvent) {
			if (((Projectile) ((ProjectileHitEvent) e).getEntity()).getShooter() instanceof Player)
				return (Player) ((Projectile) ((ProjectileHitEvent) e).getEntity()).getShooter();
		} else if (e instanceof EntityTargetEvent) {
			if (((EntityTargetEvent) e).getTarget() instanceof Player)
				return (Player) ((EntityTargetEvent) e).getTarget();
		} else if (e instanceof EntityRegainHealthEvent) {
			if (((EntityRegainHealthEvent) e).getEntity() instanceof Player)
				return (Player) ((EntityRegainHealthEvent) e).getEntity();
		} else if (e instanceof PaintingPlaceEvent) {
			return ((PaintingPlaceEvent) e).getPlayer();
		} else if (e instanceof PlayerEvent) {
			return ((PlayerEvent) e).getPlayer();
		} else if (e instanceof VehicleDamageEvent) {
			if (((VehicleDamageEvent) e).getAttacker() instanceof Player)
				return (Player) ((VehicleDamageEvent) e).getAttacker();
		} else if (e instanceof VehicleDestroyEvent) {
			if (((VehicleDestroyEvent) e).getAttacker() instanceof Player)
				return (Player) ((VehicleDestroyEvent) e).getAttacker();
		} else if (e instanceof VehicleEnterEvent) {
			if (((VehicleEnterEvent) e).getEntered() instanceof Player)
				return (Player) ((VehicleEnterEvent) e).getEntered();
		} else if (e instanceof VehicleExitEvent) {
			if (((VehicleExitEvent) e).getExited() instanceof Player)
				return (Player) ((VehicleExitEvent) e).getExited();
		} else if (e instanceof LoopEvent && ((LoopEvent) e).getCurrent() instanceof Player) {
			return (Player) ((LoopEvent) e).getCurrent();
		}
		return null;
	}
	
	public static Class<? extends Entity>[] getEntityTypes(final String[] names) {
		@SuppressWarnings("unchecked")
		final Class<? extends Entity>[] types = (Class<? extends Entity>[]) new Class<?>[names.length];
		for (int i = 0; i < names.length; i++) {
			types[i] = getEntityType(names[i]);
		}
		return types;
	}
	
	public static Class<? extends Entity> getEntityType(final String name) {
		if (name.equalsIgnoreCase("any") || name.equalsIgnoreCase("entity")) {
			return Entity.class;
		} else if (name.equalsIgnoreCase("animal")) {
			return Animals.class;
		} else if (name.equalsIgnoreCase("arrow")) {
			return Arrow.class;
		} else if (name.equalsIgnoreCase("boat")) {
			return Boat.class;
		} else if (name.equalsIgnoreCase("chicken")) {
			return Chicken.class;
		} else if (name.equalsIgnoreCase("cow")) {
			return Cow.class;
		} else if (name.equalsIgnoreCase("creature")) {
			return Creature.class;
		} else if (name.equalsIgnoreCase("creeper")) {
			return Creeper.class;
		} else if (name.equalsIgnoreCase("egg")) {
			return Egg.class;
		} else if (name.equalsIgnoreCase("explosive")) {
			return Explosive.class;
		} else if (name.equalsIgnoreCase("falling")) {
			return FallingSand.class;
		} else if (name.equalsIgnoreCase("fireball")) {
			return Fireball.class;
		} else if (name.equalsIgnoreCase("fish")) {
			return Fish.class;
		} else if (name.equalsIgnoreCase("flying")) {
			return Flying.class;
		} else if (name.equalsIgnoreCase("ghast")) {
			return Ghast.class;
		} else if (name.equalsIgnoreCase("giant")) {
			return Giant.class;
		} else if (name.equalsIgnoreCase("human")) {
			return HumanEntity.class;
		} else if (name.equalsIgnoreCase("item")) {
			return Item.class;
		} else if (name.equalsIgnoreCase("lightning strike")) {
			return LightningStrike.class;
		} else if (name.equalsIgnoreCase("living")) {
			return LivingEntity.class;
		} else if (name.equalsIgnoreCase("minecart")) {
			return Minecart.class;
		} else if (name.equalsIgnoreCase("monster")) {
			return Monster.class;
		} else if (name.equalsIgnoreCase("painting")) {
			return Painting.class;
		} else if (name.equalsIgnoreCase("pig")) {
			return Pig.class;
		} else if (name.equalsIgnoreCase("pigzombie")) {
			return PigZombie.class;
		} else if (name.equalsIgnoreCase("player")) {
			return Player.class;
		} else if (name.equalsIgnoreCase("powered minecart")) {
			return PoweredMinecart.class;
		} else if (name.equalsIgnoreCase("projectile")) {
			return Projectile.class;
		} else if (name.equalsIgnoreCase("sheep")) {
			return Sheep.class;
		} else if (name.equalsIgnoreCase("skeleton")) {
			return Skeleton.class;
		} else if (name.equalsIgnoreCase("slime")) {
			return Slime.class;
		} else if (name.equalsIgnoreCase("snowball")) {
			return Snowball.class;
		} else if (name.equalsIgnoreCase("spider")) {
			return Spider.class;
		} else if (name.equalsIgnoreCase("squid")) {
			return Squid.class;
		} else if (name.equalsIgnoreCase("storage minecart")) {
			return StorageMinecart.class;
		} else if (name.equalsIgnoreCase("tnt")) {
			return TNTPrimed.class;
		} else if (name.equalsIgnoreCase("vehicle")) {
			return Vehicle.class;
		} else if (name.equalsIgnoreCase("water")) {
			return WaterMob.class;
		} else if (name.equalsIgnoreCase("wolf")) {
			return Wolf.class;
		} else if (name.equalsIgnoreCase("zombie")) {
			return Zombie.class;
		} else if (name.equalsIgnoreCase("cave spider")) {
			return CaveSpider.class;
		} else if (name.equalsIgnoreCase("enderman")) {
			return Enderman.class;
		} else if (name.equalsIgnoreCase("experience orb")) {
			return ExperienceOrb.class;
		} else if (name.equalsIgnoreCase("silverfish")) {
			return Silverfish.class;
		}
		return null;
	}
	
	public static ItemStack[] getItemStacks(final String[] args) {
		final ItemStack[] stacks = new ItemStack[args.length];
		for (int i = 0; i < args.length; i++) {
			stacks[i] = getItemStack(args[i]);
		}
		return stacks;
	}
	
	public static ItemStack getItemStack(final String s) {
		final ItemType tt = getItemType(s);
		if (tt == null)
			return null;
		return tt.getRandom();
	}
	
	public static ItemType getItemType(final String s) {
		if (s == null || s.isEmpty())
			return null;
		final ItemType t = new ItemType();
		final String[] types = s.split("\\s*,\\s*");
		for (String type : types) {
			int amount = 1;
			if (type.matches("\\d+ .+")) {
				amount = Integer.parseInt(type.split(" ", 2)[0]);
				type = type.split(" ", 2)[1];
			}
			type = type.replace(' ', '_');
			Entry<String, ItemType> e;
			int c = type.indexOf(':');
			if (c == -1)
				c = type.length();
			final String typeR = type.substring(0, c).toLowerCase(Locale.ENGLISH);
			if ((e = containsEqualKey(Skript.aliases, typeR)) != null || (typeR.startsWith("any_") && (e = containsEqualKey(Skript.aliases, typeR.substring(4))) != null)) {
				if (c != type.length()) {
					final ItemData data = parseData(type.substring(c + 1));
					for (final ItemData d : e.getValue().types) {
						final ItemData id = d.intersection(data);
						id.amount = id.amount == -1 ? amount : id.amount * amount;
						t.types.add(id);
					}
				} else {
					for (final ItemData d : e.getValue().types) {
						d.amount = d.amount == -1 ? amount : d.amount * amount;
						t.types.add(d);
					}
				}
			} else {
				final ItemData data = getItemData(type);
				if (data == null)
					return null;
				t.types.add(data);
			}
		}
		return t;
	}
	
	/**
	 * gets the data part of an item data
	 * 
	 * @param s everything after & not including ':'
	 * @return itemdata with only the dataMin and/or dataMax set
	 */
	private final static ItemData parseData(final String s) {
		if (!s.matches("((\\d+)?-(\\d+)?)?"))
			return null;
		final ItemData t = new ItemData();
		int i = s.indexOf('-');
		if (i == -1)
			i = s.length();
		t.dataMin = (i == 0 ? -1 : parseShort(s.substring(0, i)));
		t.dataMax = (i == s.length() ? t.dataMin : (i == s.length() - 1 ? -1 : parseShort(s.substring(i + 1, s.length()))));
		return t;
	}
	
	// public static int parseRange_lower(final String s) {
	// final int i = s.indexOf('-');
	// if (i == -1)
	// return parseInt(s);
	// return i == 0 ? -1 : parseInt(s.substring(0, i));
	// }
	//
	// public static int parseRange_upper(final String s, final int lower) {
	// int i = s.indexOf('-');
	// if (i == -1)
	// return lower;
	// i = i == s.length() - 1 ? -1 : parseInt(s.substring(i + 1, s.length()));
	// if (i != -1 && i < lower)
	// Skript.error("range error: lower bound has to be lower than the upper bound");
	// return i;
	// }
	
	private static ItemData getItemData(String s) {
		if (s == null || s.isEmpty())
			return null;
		final ItemData t = new ItemData();
		if (s.matches("\\d+ .+")) {
			t.amount = Integer.parseInt(s.split(" ", 2)[0]);
			s = s.split(" ", 2)[1];
		}
		s = s.replace(' ', '_');
		final int d = s.indexOf(':');
		if (d != 0) {
			final String sx = s.substring(0, d != -1 ? d : s.length());
			final Material m = Material.matchMaterial(sx);
			if (m != null) {
				t.typeid = m.getId();
			} else {
				try {
					t.typeid = Integer.parseInt(sx);
				} catch (final NumberFormatException e1) {
					// Skript.error("invalid item type '" + s + "'");
					return null;
				}
			}
		}
		if (d != -1) {
			t.merge(parseData(s.substring(d + 1)));
		}
		return t;
	}
	
	public static Entity getEntity(final Event e) {
		if (e instanceof EntityEvent) {
			return ((EntityEvent) e).getEntity();
		} else if (e instanceof PaintingEvent) {
			return ((PaintingEvent) e).getPainting();
		} else if (e instanceof PlayerInteractEntityEvent) {
			return ((PlayerInteractEntityEvent) e).getRightClicked();
		} else if (e instanceof PlayerEvent) {
			return ((PlayerEvent) e).getPlayer();
		} else if (e instanceof VehicleEvent) {
			return ((VehicleEvent) e).getVehicle();
		}
		return null;
	}
	
	public static boolean inventoryContains(final Inventory inventory, final ItemStack[] items) {
		for (final ItemStack item : items) {
			if (!inventoryContains(inventory, item))
				return false;
		}
		return true;
	}
	
	public static boolean inventoryContains(final Inventory inventory, final ItemStack item) {
		int amount = 0;
		for (final ItemStack i : inventory.all(item.getType()).values()) {
			if (!(i.getData() == null ^ item.getData() == null) && (i.getData() == null && item.getData() == null || i.getData().getData() == item.getData().getData()))
				amount += i.getAmount();
		}
		return amount >= item.getAmount();
	}
	
	public static boolean removeItemStacks(final Inventory inventory, final ItemStack[] items, final Player player) {
		return removeItemStacks(inventory, items, player);
	}
	
	@SuppressWarnings("deprecation")
	public static boolean removeItemStacks(final Inventory inventory, final Iterable<ItemStack> items, final Player player) {
		boolean ok = true;
		for (final ItemStack item : items) {
			ok &= removeItemStack(inventory, item, null);
		}
		if (player != null) {
			player.updateInventory();
		}
		return ok;
	}
	
	@SuppressWarnings("deprecation")
	public static boolean removeItemStack(final List<ItemStack> items, final ItemStack item, final Player player) {
		int amount = item.getAmount();
		for (int i = 0; i < items.size(); i++) {
			final ItemStack is = items.get(i);
			if (is == null || is.getTypeId() != item.getTypeId())
				continue;
			if (!(is.getData() == null ^ item.getData() == null) && (is.getData() == null && item.getData() == null || is.getData().getData() == item.getData().getData())) {
				if (is.getAmount() > amount) {
					is.setAmount(is.getAmount() - amount);
					if (player != null) {
						player.updateInventory();
					}
					return true;
				}
				amount -= is.getAmount();
				items.remove(i);
				if (amount == 0) {
					if (player != null) {
						player.updateInventory();
					}
					return true;
				}
			}
		}
		return false;
	}
	
	@SuppressWarnings("deprecation")
	public static boolean removeItemStack(final Inventory inventory, final ItemStack item, final Player player) {
		int amount = item.getAmount();
		for (int i = 0; i < inventory.getSize(); i++) {
			final ItemStack is = inventory.getItem(i);
			if (is == null || is.getTypeId() != item.getTypeId())
				continue;
			if (!(is.getData() == null ^ item.getData() == null) && (is.getData() == null && item.getData() == null || is.getData().getData() == item.getData().getData())) {
				if (is.getAmount() > amount) {
					is.setAmount(is.getAmount() - amount);
					if (player != null) {
						player.updateInventory();
					}
					return true;
				}
				amount -= is.getAmount();
				inventory.clear(i);
				if (amount == 0) {
					if (player != null) {
						player.updateInventory();
					}
					return true;
				}
			}
		}
		return false;
	}
	
	public static Player getOnlinePlayer(final String name) {
		for (final Player p : Bukkit.getServer().getOnlinePlayers()) {
			if (p.getName().equalsIgnoreCase(name))
				return p;
		}
		return null;
	}
	
	public static Time parseTime(final String s) throws NumberFormatException {
		if (!s.matches("^\\d?\\d:\\d\\d$"))
			return null;
		final int c = s.indexOf(':');
		if (c == -1)
			return new Time(parseInt(s));
		return new Time((int) (parseInt(s.substring(0, c)) * 1000 - 6000 + Math.round(parseInt(s.substring(c + 1)) * 16.666667)) % 24000);
	}
	
	public static final String getTimeString(long time) {
		time = (time + 6000) % 24000;
		final int hours = (int) Math.floor(time / 1000);
		final int minutes = (int) (Math.floor(time % 1000) / 16.6667);
		return "" + hours + ":" + (minutes < 10 ? "0" : "") + minutes;
	}
	
	public static BlockState getTileEntity(final org.bukkit.block.Block b) {
		for (final BlockState bs : b.getChunk().getTileEntities()) {
			if (bs.getBlock() == b)
				return bs;
		}
		return null;
	}
	
	public static String join(final VariableString[] strings) {
		final StringBuilder b = new StringBuilder();
		for (int i = 0; i < strings.length; i++) {
			if (i != 0)
				b.append(',');
			b.append(strings[i].getDebugMessage());
		}
		return b.toString();
	}
	
	public static String join(final String[] strings) {
		return join(strings, ",");
	}
	
	public static String join(final String[] strings, final String delimiter) {
		return join(strings, delimiter, 0, strings.length);
	}
	
	public static String join(final String[] strings, final String delimiter, final int start, final int end) {
		final StringBuilder b = new StringBuilder();
		for (int i = start; i < end; i++) {
			if (i != 0)
				b.append(delimiter);
			b.append(strings[i]);
		}
		return b.toString();
	}
	
	public static String join(final Object[] objects) {
		final StringBuilder b = new StringBuilder();
		for (int i = 0; i < objects.length; i++) {
			if (i != 0)
				b.append(',');
			b.append(objects[i]);
		}
		return b.toString();
	}
	
	public static String join(final List<?> objects) {
		final StringBuilder b = new StringBuilder();
		for (int i = 0; i < objects.size(); i++) {
			if (i != 0)
				b.append(',');
			b.append(objects.get(i));
		}
		return b.toString();
	}
	
	public static Object getAttacker(final Event e) {
		if (e instanceof EntityDamageByEntityEvent) {
			if (((EntityDamageByEntityEvent) e).getDamager() instanceof Projectile) {
				return ((Projectile) ((EntityDamageByEntityEvent) e).getDamager()).getShooter();
			}
			return ((EntityDamageByEntityEvent) e).getDamager();
		} else if (e instanceof EntityDamageByBlockEvent) {
			return ((EntityDamageByBlockEvent) e).getDamager();
		}
		return null;
	}
	
	public static final String fancyClassName(final String className) {
		final StringBuilder r = new StringBuilder(className.length() + 5);
		
		return r.toString();
	}
	
	public static <T> T getRandom(final T[] os) {
		return os[Skript.random.nextInt(os.length)];
	}
	
	public static <T> T getRandom(final List<T> os) {
		return os.get(Skript.random.nextInt(os.size()));
	}
	
	/**
	 * tests whether two item stacks are of the same type, i.e. it ignores the amounts.
	 * 
	 * @param is1
	 * @param is2
	 * @return
	 */
	public static boolean itemStacksEqual(final ItemStack is1, final ItemStack is2) {
		return is1.getTypeId() == is2.getTypeId() && is1.getData().getData() == is2.getData().getData() && is1.getDurability() == is2.getDurability();
	}
	
	public static String replaceAll(final String string, final String pattern, final Callback<String, Matcher> callback) {
		final Matcher m = Pattern.compile(pattern).matcher(string);
		final StringBuffer b = new StringBuffer();
		while (m.find()) {
			m.appendReplacement(b, callback.run(m));
		}
		m.appendTail(b);
		return b.toString();
	}
	
	/**
	 * note: changes 'from'
	 * 
	 * @param what
	 * @param from
	 * @return from
	 */
	public static final ItemStack remove(final ItemStack what, final ItemStack from) {
		if (what == null || from == null || !itemStacksEqual(what, from))
			return from;
		from.setAmount(Math.max(from.getAmount() - what.getAmount(), 0));
		return from;
	}
	
	/**
	 * note: changes 'to'
	 * 
	 * @param what
	 * @param to
	 * @return to
	 */
	public static final ItemStack add(final ItemStack what, final ItemStack to) {
		if (what == null || to == null || !itemStacksEqual(what, to))
			return to;
		to.setAmount(Math.max(to.getAmount() + what.getAmount(), 0));
		return to;
	}
	
	public static Player getTargetPlayer(final Player player) {
		return getTarget(player, player.getWorld().getPlayers(), Player.class);
	}
	
	public static Entity getTargetEntity(final Entity entity, Class<? extends Entity> type) {
		return getTarget(entity, entity.getWorld().getEntities(), type);
	}
	
	public static <T extends Entity> T getTarget(final Entity entity, final Iterable<T> entities, Class<? extends Entity> type) {
		if (entity == null)
			return null;
		T target = null;
		final double radiusSquared = 1;
		final Vector l = entity.getLocation().toVector();
		if (entity instanceof LivingEntity)
			l.setY(l.getY() + ((LivingEntity) entity).getEyeHeight());
		for (final T other : entities) {
			if (!(type.isInstance(other)))
				continue;
			final Vector n = other.getLocation().toVector().subtract(l);
			if (entity.getLocation().getDirection().normalize().crossProduct(n).lengthSquared() < radiusSquared && n.normalize().dot(entity.getLocation().getDirection().normalize()) >= 0) {
				if (target == null || target.getLocation().distanceSquared(entity.getLocation()) > other.getLocation().distanceSquared(entity.getLocation()))
					target = other;
			}
		}
		return target;
	}
	
	/**
	 * Tests whether an inventory has enough space to hold the given item stacks. The given stacks are assumed to be stacked as much as possible.
	 * 
	 * @param i
	 * @param items
	 * @return
	 */
	public static boolean canHold(Inventory invi, ItemStack[] items) {
		final Integer[] added = new Integer[items.length];
		final Integer[] contained = new Integer[invi.getSize()];
		int numFreeSlots = 0;
		for (int i = 0; i < items.length; i++) {
			for (int j = invi.getSize() - 1; j >= 0; j--) {// inventories are usually filled from the beginning, so empty slots are at the end
				ItemStack c = invi.getItem(j);
				if (c == null || c.getAmount() == 0) {
					numFreeSlots++;
					if (numFreeSlots >= items.length) // performance increaser for more or less empty inventories
						return true;
					continue;
				}
				if (Utils.itemStacksEqual(c, items[i])) {
					final int spaceLeft = c.getMaxStackSize() - c.getAmount() - contained[j];
					if (spaceLeft >= items[i].getAmount()) {
						added[i] = items[i].getAmount();
						contained[j] += items[i].getAmount();
						break;
					} else {
						int t = items[i].getAmount() - spaceLeft;
						added[i] += t;
						contained[j] += t;
					}
				}
			}
		}
		for (int i = 0; i < items.length; i++) {
			if (added[i] != items[i].getAmount()) {
				if (numFreeSlots == 0)
					return false;
				numFreeSlots--;
			}
		}
		return true;
	}
	
	public static List<Player> getPlayersInCone(Player player, double coneAngle, double radius) {
		final double coneArea = Math.tan(coneAngle)*Math.tan(coneAngle), radiusSquared = radius*radius;
		ArrayList<Player> targets = new ArrayList<Player>();
		for (Player other:player.getWorld().getPlayers()) {
			Vector n = other.getLocation().toVector().subtract(player.getLocation().toVector()).normalize();
			if (player.getLocation().getDirection().normalize().crossProduct(n).lengthSquared() <= coneArea // within cone
					&& player.getLocation().distanceSquared(other.getLocation()) <= radiusSquared // within radius
					&& player.getLocation().getDirection().dot(n) >= 0) // same direction
				targets.add(other);
		}
		return targets;
	}
	
}
