/*
 * This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 2.5 Switzerland License. To view a copy of this license, visit
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ch/ or send a letter to Creative Commons, 444 Castro Street, Suite 900, Mountain View, California, 94041, USA.
 */

package ch.njol.skript.data;

import java.util.Locale;

import org.bukkit.Bukkit;
import org.bukkit.Location;
import org.bukkit.TreeType;
import org.bukkit.World;
import org.bukkit.block.Block;
import org.bukkit.block.BlockFace;
import org.bukkit.command.CommandSender;
import org.bukkit.entity.Entity;
import org.bukkit.entity.Player;
import org.bukkit.event.Event;
import org.bukkit.event.player.PlayerCommandPreprocessEvent;
import org.bukkit.event.server.ServerCommandEvent;
import org.bukkit.inventory.Inventory;
import org.bukkit.inventory.ItemStack;

import ch.njol.skript.Skript;
import ch.njol.skript.api.Literal;
import ch.njol.skript.api.Parser;
import ch.njol.skript.api.SimpleDefaultVariable;
import ch.njol.skript.util.EntityType;
import ch.njol.skript.util.ItemData;
import ch.njol.skript.util.ItemType;
import ch.njol.skript.util.Time;
import ch.njol.skript.util.TimePeriod;
import ch.njol.skript.util.Timespan;
import ch.njol.skript.util.Utils;
import ch.njol.skript.util.VariableString;
import ch.njol.skript.util.WeatherType;

/**
 * @author Peter Gttinger
 * 
 */
public class DefaultClasses {
	
	public DefaultClasses() {}
	
	static {
		Skript.addClass("object", Object.class, null, null);
	}
	
	public static final class FloatDefaultVariable extends Literal<Float> {
		public FloatDefaultVariable() {
			super(1f);
		}
	}
	
	static {
		Skript.addClass("float", Float.class, FloatDefaultVariable.class, new Parser<Float>() {
			@Override
			public Float parse(final String s) {
				try {
					if (s.endsWith("%")) {
						return Float.valueOf(s.substring(0, s.length() - 1)) / 100;
					}
					return Float.valueOf(s);
				} catch (final NumberFormatException e) {
					return null;
				}
			}
			
			@Override
			public String toString(final Float o) {
				return o.toString();
			}
		});
	}
	
	public static final class DoubleDefaultVariable extends Literal<Double> {
		public DoubleDefaultVariable() {
			super(1.);
		}
	}
	
	static {
		Skript.addClass("double", Double.class, DoubleDefaultVariable.class, new Parser<Double>() {
			@Override
			public Double parse(final String s) {
				try {
					if (s.endsWith("%")) {
						return Double.valueOf(s.substring(0, s.length() - 1)) / 100;
					}
					return Double.valueOf(s);
				} catch (final NumberFormatException e) {
					return null;
				}
			}
			
			@Override
			public String toString(final Double o) {
				return o.toString();
			}
		}, "numbers?");
	}
	
	public static final class BooleanDefaultVariable extends Literal<Boolean> {
		public BooleanDefaultVariable() {
			super(true);
		}
	}
	
	static {
		Skript.addClass("boolean", Boolean.class, BooleanDefaultVariable.class, new Parser<Boolean>() {
			@Override
			public Boolean parse(final String s) {
				final int i = Utils.parseBooleanNoError(s);
				if (i == 1)
					return Boolean.TRUE;
				if (i == 0)
					return Boolean.FALSE;
				return null;
			}
			
			@Override
			public String toString(final Boolean o) {
				return o.toString();
			}
		});
	}
	
	public static final class ByteDefaultVariable extends Literal<Byte> {
		public ByteDefaultVariable() {
			super((byte) 1);
		}
	}
	
	static {
		Skript.addClass("byte", Byte.class, ByteDefaultVariable.class, new Parser<Byte>() {
			@Override
			public Byte parse(final String s) {
				try {
					return Byte.valueOf(s);
				} catch (final NumberFormatException e) {
					return null;
				}
			}
			
			@Override
			public String toString(final Byte o) {
				return o.toString();
			}
		});
	}
	
	public static final class ShortDefaultVariable extends Literal<Short> {
		public ShortDefaultVariable() {
			super((short) 1);
		}
	}
	
	static {
		Skript.addClass("short", Short.class, ShortDefaultVariable.class, new Parser<Short>() {
			@Override
			public Short parse(final String s) {
				try {
					return Short.valueOf(s);
				} catch (final NumberFormatException e) {
					return null;
				}
			}
			
			@Override
			public String toString(final Short o) {
				return o.toString();
			}
		});
	}
	
	public static final class IntegerDefaultVariable extends Literal<Integer> {
		public IntegerDefaultVariable() {
			super(1);
		}
	}
	
	static {
		Skript.addClass("integer", Integer.class, IntegerDefaultVariable.class, new Parser<Integer>() {
			@Override
			public Integer parse(final String s) {
				try {
					return Integer.valueOf(s);
				} catch (final NumberFormatException e) {
					return null;
				}
			}
			
			@Override
			public String toString(final Integer o) {
				return o.toString();
			}
		}, "integers?");
	}
	
	public static final class LongDefaultVariable extends Literal<Long> {
		public LongDefaultVariable() {
			super(1L);
		}
	}
	
	static {
		Skript.addClass("long", Long.class, LongDefaultVariable.class, new Parser<Long>() {
			@Override
			public Long parse(final String s) {
				try {
					return Long.valueOf(s);
				} catch (final NumberFormatException e) {
					return null;
				}
			}
			
			@Override
			public String toString(final Long o) {
				return o.toString();
			}
		});
	}
	
	static {
		Skript.addClass("string", String.class, null, new Parser<String>() {
			@Override
			public String parse(final String s) {
				if (!s.startsWith("\"") || !s.endsWith("\""))
					return null;
				return s.substring(1, s.length() - 1);
			}
			
			@Override
			public String toString(final String s) {
				return s;
			}
		});
	}
	
	static {
		Skript.addClass("variablestring", VariableString.class, null, new Parser<VariableString>() {
			
			@Override
			public VariableString parse(final String s) {
				if (!s.startsWith("\"") || !s.endsWith("\""))
					return null;
				return new VariableString(s.substring(1, s.length() - 1));
			}
			
			@Override
			public String toString(final VariableString o) {
				return "";
			}
			
		});
	}
	
	public static final class ItemStackDefaultVariable extends SimpleDefaultVariable<ItemStack> {
		
		@Override
		public ItemStack[] getAll(final Event e) {
			return new ItemStack[] {Skript.getEventValue(e, ItemStack.class)};
		}
		
	}
	
	static {
		Skript.addClass("itemstack", ItemStack.class, ItemStackDefaultVariable.class, new Parser<ItemStack>() {
			@Override
			public ItemStack parse(final String s) {
				return Utils.getItemStack(s);
			}
			
			@Override
			public String toString(final ItemStack o) {
				return o.getTypeId() + (o.getDurability() != 0 ? ":" + o.getDurability() : (o.getData() != null ? ":" + o.getData().getData() : "")) + "*" + o.getAmount();
			}
		});
	}
	
	public static final class ItemTypeDefaultVariable extends Literal<ItemType> {
		public ItemTypeDefaultVariable() {
			super(new ItemType(new ItemData()));
		}
	}
	
	static {
		Skript.addClass("itemtype", ItemType.class, ItemTypeDefaultVariable.class, new Parser<ItemType>() {
			@Override
			public ItemType parse(final String s) {
				return Utils.getItemType(s);
			}
			
			@Override
			public String toString(final ItemType o) {
				return o.toString();
			}
		}, "item ?types?", "items?");
	}
	
	public static final class TimeDefaultVariable extends SimpleDefaultVariable<Time> {
		
		@Override
		public Time[] getAll(final Event e) {
			final World w = Skript.getEventValue(e, World.class);
			if (w == null)
				return null;
			return new Time[] {new Time((int) w.getTime())};
		}
		
	}
	
	static {
		Skript.addClass("time", Time.class, TimeDefaultVariable.class, new Parser<Time>() {
			
			@Override
			public Time parse(final String s) {
				return Utils.parseTime(s);
			}
			
			@Override
			public String toString(final Time o) {
				return Utils.getTimeString(o.get());
			}
			
		}, "times?");
	}
	
	static {
		new Timespan();
	}
	
	public static final class TimePeriodDefaultVariable extends Literal<TimePeriod> {
		public TimePeriodDefaultVariable() {
			super(new TimePeriod(0, 23999));
		}
	}
	
	static {
		Skript.addClass("timeperiod", TimePeriod.class, TimePeriodDefaultVariable.class, new Parser<TimePeriod>() {
			@Override
			public TimePeriod parse(final String s) {
				if (s.equalsIgnoreCase("day")) {
					return new TimePeriod(0, 11999);
				} else if (s.equalsIgnoreCase("dusk")) {
					return new TimePeriod(12000, 13799);
				} else if (s.equalsIgnoreCase("night")) {
					return new TimePeriod(13800, 22199);
				} else if (s.equalsIgnoreCase("dawn")) {
					return new TimePeriod(22200, 23999);
				}
				final int c = s.indexOf('-');
				try {
					if (c == -1) {
						final Time t = Utils.parseTime(s);
						if (t == null)
							return null;
						return new TimePeriod(t.get());
					}
					final Time t1 = Utils.parseTime(s.substring(0, c));
					final Time t2 = Utils.parseTime(s.substring(c + 1));
					if (t1 == null || t2 == null)
						return null;
					return new TimePeriod(t1.get(), t2.get());
				} catch (final NumberFormatException e) {
					return null;
				}
			}
			
			@Override
			public String toString(final TimePeriod o) {
				return o.toString();
			}
		}, "time ?periods?", "durations?");
	}
	
	public static final class TreeTypeDefaultVariable extends Literal<TreeType> {
		public TreeTypeDefaultVariable() {
			super(TreeType.TREE);
		}
	}
	
	static {
		Skript.addClass("treetype", TreeType.class, TreeTypeDefaultVariable.class, new Parser<TreeType>() {
			
			@Override
			public TreeType parse(final String s) {
				try {
					return TreeType.valueOf(s.toUpperCase(Locale.ENGLISH).replace(' ', '_'));
				} catch (final IllegalArgumentException e) {
					return null;
				}
			}
			
			@Override
			public String toString(final TreeType o) {
				return o.toString().toLowerCase().replace('_', ' ');
			}
			
		}, "tree ?types?", "trees?");
	}
	
	public static final class WeatherTypeDefaultVariable extends Literal<WeatherType> {
		public WeatherTypeDefaultVariable() {
			super(WeatherType.clear);
		}
	}
	
	static {
		Skript.addClass("weathertype", WeatherType.class, WeatherTypeDefaultVariable.class, new Parser<WeatherType>() {
			
			@Override
			public WeatherType parse(final String s) {
				return WeatherType.parse(s);
			}
			
			@Override
			public String toString(final WeatherType o) {
				return o.toString();
			}
			
		}, "weather ?types?", "weather conditions", "weathers?");
	}
	
	public static final class EntityTypeDefaultVariable extends Literal<EntityType> {
		public EntityTypeDefaultVariable() {
			super(new EntityType(Entity.class, 1));
		}
	}
	
	static {
		Skript.addClass("entitytype", EntityType.class, EntityTypeDefaultVariable.class, new Parser<EntityType>() {
			@Override
			public EntityType parse(final String s) {
				if (s.matches("\\d+ .+")) {
					return new EntityType(Utils.getEntityType(s.split(" ", 2)[1]), Integer.parseInt(s.split(" ", 2)[0]));
				}
				return new EntityType(Utils.getEntityType(s), 1);
			}
			
			@Override
			public String toString(final EntityType o) {
				return o.toString();
			}
		}, "entity ?types?", "enit(y|ies)");
	}
	
	public static final class BlockDefaultVariable extends SimpleDefaultVariable<Block> {
		
		@Override
		public Block[] getAll(final Event e) {
			return new Block[] {Skript.getEventValue(e, Block.class)};
		}
		
	}
	
	static {
		Skript.addClass("block", Block.class, BlockDefaultVariable.class, null);
	}
	
	public static final class LocationDefaultVariable extends SimpleDefaultVariable<Location> {
		
		@Override
		public Location[] getAll(final Event e) {
			return new Location[] {Skript.getEventValue(e, Location.class)};
		}
		
	}
	
	static {
		Skript.addClass("location", Location.class, LocationDefaultVariable.class, null);
	}
	
	public static final class WorldDefaultVariable extends SimpleDefaultVariable<World> {
		
		@Override
		public World[] getAll(final Event e) {
			return new World[] {Skript.getEventValue(e, World.class)};
		}
		
	}
	
	static {
		Skript.addClass("world", World.class, WorldDefaultVariable.class, new Parser<World>() {
			@Override
			public World parse(final String s) {
				return Bukkit.getServer().getWorld(s);
			}
			
			@Override
			public String toString(final World o) {
				return o.getName();
			}
		}, "worlds?");
	}
	
	public static final class CommandSenderDefaultVariable extends SimpleDefaultVariable<CommandSender> {
		
		@Override
		public CommandSender[] getAll(final Event e) {
			if (e instanceof ServerCommandEvent) {
				return new CommandSender[] {((ServerCommandEvent) e).getSender()};
			} else if (e instanceof PlayerCommandPreprocessEvent) {
				return new CommandSender[] {((PlayerCommandPreprocessEvent) e).getPlayer()};
			}
			return null;
		}
		
	}
	
	static {
		Skript.addClass("commandsender", CommandSender.class, CommandSenderDefaultVariable.class, new Parser<CommandSender>() {
			@Override
			public CommandSender parse(final String s) {
				if (s.equalsIgnoreCase("console"))
					return Bukkit.getConsoleSender();
				return Bukkit.getServer().getPlayerExact(s);
			}
			
			@Override
			public String toString(final CommandSender o) {
				if (o instanceof Player)
					return ((Player) o).getDisplayName();
				return o.getName();
			}
		});
	}
	
	public static final class InventoryDefaultVariable extends SimpleDefaultVariable<Inventory> {
		
		@Override
		public Inventory[] getAll(final Event e) {
			return new Inventory[] {Skript.getEventValue(e, Inventory.class)};
		}
		
	}
	
	static {
		Skript.addClass("inventory", Inventory.class, InventoryDefaultVariable.class, null);
	}
	
	public static final class PlayerDefaultVariable extends SimpleDefaultVariable<Player> {
		
		@Override
		public Player[] getAll(final Event e) {
			return new Player[] {Skript.getEventValue(e, Player.class)};
		}
		
	}
	
	static {
		Skript.addClass("player", Player.class, PlayerDefaultVariable.class, new Parser<Player>() {
			@Override
			public Player parse(final String s) {
				return Bukkit.getServer().getPlayerExact(s);
			}
			
			@Override
			public String toString(final Player o) {
				return o.getDisplayName();
			}
		}, "players?");
	}
	
	public static final class BlockFaceDefaultVariable extends SimpleDefaultVariable<BlockFace> {
		
		@Override
		public BlockFace[] getAll(final Event e) {
			return new BlockFace[] {Skript.getEventValue(e, BlockFace.class)};
		}
		
	}
	
	static {
		Skript.addClass("direction", BlockFace.class, BlockFaceDefaultVariable.class, new Parser<BlockFace>() {
			@Override
			public BlockFace parse(final String s) {
				return Utils.getBlockFace(s, true);
			}
			
			@Override
			public String toString(final BlockFace o) {
				return o.toString().toLowerCase(Locale.ENGLISH).replace('_', ' ');
			}
		}, "directions?");
		
	}
}
