/*
 * This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 2.5 Switzerland License. To view a copy of this license, visit
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ch/ or send a letter to Creative Commons, 444 Castro Street, Suite 900, Mountain View, California, 94041, USA.
 */

package ch.njol.skript.config;

import java.io.PrintWriter;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import ch.njol.skript.Skript;
import ch.njol.skript.SkriptLogger;

public abstract class ConfigNode {
	
	protected String name;
	
	protected final int line;
	/**
	 * "original" String. This is not final as VoidNodes directly change this if they are modified.
	 */
	protected String orig;
	protected boolean modified = false;
	
	protected GroupNode parent;
	protected final Config config;
	
	/**
	 * creates an empty node
	 * 
	 * @param name
	 * @param parent
	 */
	protected ConfigNode(final String name, final GroupNode parent) {
		this.name = name;
		orig = parent.getConfig().r.getLine();
		line = parent.getConfig().r.getLineNum();
		config = parent.getConfig();
		this.parent = parent;
		SkriptLogger.setNode(this);
	}
	
	/**
	 * main node constructor.<br/>
	 * never use except in {@link GroupNode#ConfigGroup(Config) new ConfigGroup(Config)}
	 * 
	 * @param c
	 */
	protected ConfigNode(final Config c) {
		name = null;
		orig = null;
		line = -1;
		config = c;
		parent = null;
		SkriptLogger.setNode(this);
	}
	
	void setIndentation(final String option) {
		if (option.equals("tab") || option.equals("tabs")) {
			config.indentation = "\t";
			config.indentationName = "tab";
		} else if (option.equals("spaces")) {
			config.indentation = "    ";
			config.indentationName = "4 spaces";
		} else if (option.equals("space")) {
			config.indentation = " ";
			config.indentationName = option;
		} else if (option.matches("[1-9] spaces")) {
			final int i = Integer.parseInt(option.substring(0, option.indexOf(' ')));
			config.indentationName = option;
			config.indentation = " ";
			for (int j = 1; j < i; j++)
				config.indentation += " ";
		} else {
			Skript.error("unknown indentation type");
		}
	}
	
	public String getName() {
		return name;
	}
	
	public Config getConfig() {
		return config;
	}
	
	public void rename(final String newname) {
		if (name == null) {
			Skript.error("can't rename an anonymous node!");
			return;
		}
		name = newname;
		modified();
	}
	
	public void move(final GroupNode newParent) {
		if (parent == null) {
			Skript.error("can't move the main node!");
			return;
		}
		if (!newParent.isGroup()) {
			Skript.error("non-group nodes can't have subnodes!");
			return;
		}
		parent.getNodeList().remove(this);
		parent = newParent;
		newParent.getNodeList().add(this);
		config.modified = true;
	}
	
	protected void modified() {
		modified = true;
		config.modified = true;
	}
	
	protected String getComment() {
		final Matcher m = Pattern.compile("\\s*(?<!#)#[^#].*$").matcher(getOrig());
		if (!m.find())
			return "";
		return m.group();
	}
	
	protected String getIndentation() {
		String s = "";
		ConfigNode n = this;
		while ((n = n.parent) != null) {
			s += config.indentation;
		}
		return s;
	}
	
	abstract void save(final PrintWriter w);
	
	public GroupNode getParent() {
		return parent;
	}
	
	public void move(final int index) {
		parent.getNodeList().remove(this);
		parent.getNodeList().add(index, this);
		config.modified = true;
	}
	
	public void moveDelta(final int deltaIndex) {
		move(parent.getNodeList().indexOf(this) + deltaIndex);
	}
	
	public boolean setValue(final String v) {
		return false;
	}
	
	public void delete() {
		parent.getNodeList().remove(this);
		config.modified = true;
	}
	
	public String getOrig() {
		return orig;
	}
	
	public boolean isGroup() {
		return this instanceof GroupNode;
	}
	
	/**
	 * void here = void (incl. invalid) + parse option
	 * 
	 * @return
	 */
	public boolean isVoid() {
		return this instanceof VoidNode || isParseOption();
	}
	
	public boolean isInvalid() {
		return this instanceof InvalidNode;
	}
	
	public boolean isEntry() {
		return this instanceof EntryNode || this instanceof SimpleNode;
	}
	
	private boolean isParseOption() {
		return this instanceof ParseOptionNode;
	}
	
	public int getLine() {
		return line;
	}
	
	/**
	 * get a node via path:to:the:node. relative paths are possible by starting with a ':'; a double colon '::' will go up a node.<br/>
	 * selecting the n-th node can be done with #n.
	 * 
	 * @param path
	 * @return the node at the given path or null if the path is invalid
	 */
	public ConfigNode getNode(final String path) {
		return getNode(path, false);
	}
	
	public ConfigNode getNode(String path, final boolean create) {
		ConfigNode n;
		if (path.startsWith(":")) {
			path = path.substring(1);
			n = this;
		} else {
			n = config.getMainNode();
		}
		for (final String s : path.split(":")) {
			if (s.isEmpty()) {
				n = n.getParent();
				if (n == null) {
					n = config.getMainNode();
				}
				continue;
			}
			if (!n.isGroup()) {
				return null;
			}
			if (s.startsWith("#")) {
				int i = -1;
				try {
					i = Integer.parseInt(s.substring(1));
				} catch (final NumberFormatException e) {
					return null;
				}
				if (i <= 0 || i > ((GroupNode) n).getNodeList().size())
					return null;
				n = ((GroupNode) n).getNodeList().get(i - 1);
			} else {
				final ConfigNode oldn = n;
				n = ((GroupNode) n).get(s);
				if (n == null) {
					if (!create)
						return null;
					((GroupNode) oldn).getNodeList().add(n = new GroupNode(s, (GroupNode) oldn));
				}
			}
		}
		return n;
	}
	
}
