/*
 * This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 2.5 Switzerland License. To view a copy of this license, visit
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ch/ or send a letter to Creative Commons, 444 Castro Street, Suite 900, Mountain View, California, 94041, USA.
 */

package ch.njol.skript.api;

import java.lang.reflect.Array;
import java.util.Iterator;
import java.util.ListIterator;

import org.bukkit.event.Event;

import ch.njol.skript.Skript;
import ch.njol.skript.util.Utils;
import ch.njol.util.Checker;

/**
 * 
 * A variable.
 * 
 * @author Peter Gttinger
 * 
 */
public abstract class Variable<T> implements Expression {
	
	protected boolean and;
	
	protected Variable() {}
	
	/**
	 * DO NOT USE THIS IN CONDITIONS, use check() instead.
	 * 
	 * @param e
	 * @return may return null
	 */
	@SuppressWarnings("unchecked")
	public final T[] get(final Event e) {
		final T[] all = getAll(e);
		if (all == null)
			return null;
		if (and || all.length <= 1)
			return all;
		final T r = Utils.getRandom(all);
		if (r == null)
			return null;
		final T[] one = (T[]) Array.newInstance(r.getClass(), 1);
		one[0] = r;
		return one;
	}
	
	protected abstract T[] getAll(Event e);
	
	/**
	 * Checks through the values to find you whether this variable matches the checker.
	 * 
	 * The argument of the checker is guaranteed to never be null.
	 * 
	 * @param e
	 * @param c
	 * @return
	 */
	public final boolean check(final Event e, final Checker<T> c, Condition cond) {
		return check(e, c, cond.getInverted());
	}
	public final boolean check(final Event e, final Checker<T> c, boolean invert) {
		for (final T o : getAll(e)) {
			final boolean b = (o == null ? false : c.check(o));
			if (and && !b)
				return false;
			if (!and && b)
				return true;
		}
		return and;
	}
	
	public abstract void set(Event e, Object[] to);
	
	public abstract void add(Event e, Object[] add);
	
	public abstract void remove(Event e, Object[] rem);
	
	public abstract void clear(Event e);

	protected <R> ConvertedVariable<? extends T, R> getConvertedVar(Class<R> to) {
		return ConvertedVariable.newInstance(this, to);
	}
	
	public static final <T> Variable<T> parse(final String s, final Class<T> returnType) {
		return parse(s, (Class<T>) returnType, Skript.getVariables().listIterator());
	}

	public static final Variable<?> parseNoLiteral(final String s, final Iterator<? extends VariableInfo<?>> source) {
		final Variable<?> var = (Variable<?>) Expressions.parse(s, source);
		
		if (var != null) {
			Skript.clearWithheldError();
			return var;
		}
		
		return null;
	}
	
	public static final <T> Variable<T> parse(final String s, final Class<T> returnType, final ListIterator<? extends VariableInfo<?>> source) {

		Variable<?> v = parseNoLiteral(s, source);
		if (v != null)
			return v.getConvertedVar(returnType);
		
		Literal<T> t = Expressions.parseLiteralArray(s, returnType);
		if (t != null) {
			Skript.clearWithheldError();
			return t;
		}
		
		Skript.error("'" + s + "' is not a " + returnType.getSimpleName(), true);
		return null;
	}
	
	/**
	 * this is set automatically and should not be changed.
	 * 
	 * @param and
	 */
	public void setAnd(final boolean and) {
		this.and = and;
	}
	
}
