/*
 * This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 2.5 Switzerland License. To view a copy of this license, visit
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ch/ or send a letter to Creative Commons, 444 Castro Street, Suite 900, Mountain View, California, 94041, USA.
 */

package ch.njol.skript.api;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import ch.njol.skript.Skript;
import ch.njol.skript.api.SkriptEvent.SkriptEventInfo;
import ch.njol.skript.util.Utils;
import ch.njol.util.Callback;

/**
 * @author Peter Gttinger
 * 
 */
public interface Expression {
	
	public static abstract class ExpressionInfo {
		
		public Class<? extends Expression> c;
		
		public Pattern[] patterns;
		public String[][] groups;
		
		public ExpressionInfo(final String[] patterns, final Class<? extends Expression> c) {
			this.patterns = new Pattern[patterns.length];
			this.groups = new String[patterns.length][];
			for (int i = 0; i < patterns.length; i++) {
				final ArrayList<String> groups = new ArrayList<String>();
				this.patterns[i] = Pattern.compile("^" + Utils.replaceAll(patterns[i], "%(.+?)%", new Callback<String, Matcher>() {
					private int num = 0;
					
					@Override
					public String run(final Matcher m) {
						num++;
						groups.add(m.group(1) + num);
						return "(?<" + m.group(1) + num + ">.+?)";
					}
				}) + "$");
				this.groups[i] = groups.toArray(new String[0]);
			}
			this.c = c;
		}
	}
	
	public static class VariableInfo<T> extends ExpressionInfo {
		
		public Class<T> returnType;
		
		public VariableInfo(final String[] patterns, final Class<T> returnType, final Class<? extends Expression> c) {
			super(patterns, c);
			this.returnType = returnType;
		}
	}
	
	public static final class Expressions {
		
		static final Expression parse(final String s, final Iterator<? extends ExpressionInfo> source) {
			Skript.clearWithheldError();
			while (source.hasNext()) {
				final Expression e = parse_i(s, source.next());
				if (e != null)
					return e;
			}
			// no error message here!
			return null;
		}

		private final static Expression parse_i(final String s, final ExpressionInfo info) {
			for (int i = 0; i < info.patterns.length; i++) {
				final Matcher m = info.patterns[i].matcher(s);
				if (m.matches()) {
					final ArrayList<Variable<?>> parts = new ArrayList<Variable<?>>(info.groups.length);
					for (final String t : info.groups[i]) {
						final String g = m.group(t);
						if (g == null) {
							try {
								final Class<? extends Variable<?>> dv = Skript.getDefaultVariable(t.replaceFirst("\\d+$", ""));
								parts.add(dv == null ? null : dv.newInstance());
							} catch (final InstantiationException e) {
								e.printStackTrace();
							} catch (final IllegalAccessException e) {
								e.printStackTrace();
							}
						} else {
							final Variable<?> p = Variable.parse(g, Skript.getClass(t.replaceFirst("\\d+$", "")));
							if (p == null)
								return null;
							parts.add(p);
						}
					}
					try {
						final Expression p = info.c.newInstance();
						p.init(parts, i, m);
						Skript.clearWithheldError();
						return p;
					} catch (final InstantiationException e) {
						e.printStackTrace();
					} catch (final IllegalAccessException e) {
						e.printStackTrace();
					}
					return null;
				}
			}
			return null;
		}
		
		static final SkriptEvent parseEvent(String s) {
			s = s.replaceFirst("^on ", "");
			for (final SkriptEventInfo info : Skript.getEvents()) {
				for (int i = 0; i < info.patterns.length; i++) {
					final Matcher m = info.patterns[i].matcher(s);
					if (m.matches()) {
						final Object[][] parts = new Object[info.groups[i].length][];
						int j = -1;
						for (final String t : info.groups[i]) {
							j++;
							final String g = m.group(t);
							if (g == null) {
								parts[j] = null;
							} else {
								final Object[] p = parseDataArray(g, Skript.getClass(t.replaceFirst("\\d+$", "")));
								if (p == null)
									return null;
								parts[j] = p;
							}
						}
						try {
							final SkriptEvent e = info.c.newInstance();
							e.init(parts, i, m);
							Skript.clearWithheldError();
							return e;
						} catch (final InstantiationException e) {
							e.printStackTrace();
						} catch (final IllegalAccessException e) {
							e.printStackTrace();
						}
						return null;
					}
				}
			}
			return null;
		}
		
		protected final static <T> Literal<T> parseLiteralArray(final String s, final Class<T> c) {
			final boolean and = !s.matches(" or ");
			final T[] ts = parseDataArray(s, c);
			if (ts == null)
				return null;
			Skript.clearWithheldError();
			return new Literal<T>(ts, and);
		}
		
		@SuppressWarnings("unchecked")
		public final static <T> T[] parseDataArray(final String s, final Class<T> c) {
			final boolean and = !s.matches(" or ");
			final String[] p = s.split(and ? ",|,? and " : ",|,? or ");
			final T[] data = (T[]) Array.newInstance(c, p.length);
			for (int i = 0; i < p.length; i++) {
				final T t = Skript.parse(p[i].trim(), c);
				if (t == null) {
					Skript.error("'" + p[i] + "' is not a " + c.getSimpleName(), true);
					return null;
				}
				data[i] = t;
			}
			Skript.clearWithheldError();
			
			return data;
		}
		
	}
	
	/**
	 * called just after the constructor.
	 * 
	 * @param vars all %var%s included in the matching pattern in the order they appear in the pattern. If an optional value was left out it will still be included in this list
	 *            holding the default value of the desired type which usually depends on the event.
	 * @param matchedPattern the pattern which matched
	 * @param matcher all info about the match. Note that any (Nth) %variable% is converted to (?&lt;variableN&gt;.+?) and is thus a group which has an index.
	 */
	public void init(List<Variable<?>> vars, int matchedPattern, Matcher matcher);
	
}
