/*
 * This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 2.5 Switzerland License. To view a copy of this license, visit
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ch/ or send a letter to Creative Commons, 444 Castro Street, Suite 900, Mountain View, California, 94041, USA.
 */

package ch.njol.skript;

import java.util.ArrayList;

import org.bukkit.event.Event;

import ch.njol.skript.api.Loop;
import ch.njol.skript.api.LoopVar;
import ch.njol.skript.api.SkriptEvent;
import ch.njol.skript.api.TopLevelExpression;
import ch.njol.skript.api.TriggerItem;
import ch.njol.skript.api.Variable;
import ch.njol.skript.config.Config;
import ch.njol.skript.config.ConfigNode;
import ch.njol.skript.config.GroupNode;
import ch.njol.skript.config.SimpleNode;
import ch.njol.util.Pair;

/**
 * 
 * @author Peter Gttinger
 * 
 */
final public class TriggerFileLoader {
	private TriggerFileLoader() {}
	
	public static ArrayList<Pair<String, LoopVar<?>>> currentLoops = new ArrayList<Pair<String, LoopVar<?>>>();
	
	@SuppressWarnings("unchecked")
	private static ArrayList<TriggerItem> loadItems(final GroupNode node) {
		final ArrayList<TriggerItem> items = new ArrayList<TriggerItem>();
		for (final ConfigNode n : node) {
			if (n.isVoid())
				continue;
			SkriptLogger.setNode(n);
			if (n.isEntry()) {
				final SimpleNode e = (SimpleNode) n;
				
				final TopLevelExpression expr = TopLevelExpression.parse(e.getName());
				if (expr == null) {
					Skript.printWithheldError("Skript can't understand this condition/effect: '" + e.getName() + "'");
					continue;
				}
				items.add(expr);
			} else if (n.isGroup()) {
				if (n.getName().startsWith("loop ")) {
					final LoopVar<?> loopvar = (LoopVar<?>) Variable.parseNoLiteral(n.getName().substring("loop ".length()), Skript.loops.listIterator());
					if (loopvar == null) {
						Skript.printWithheldError("Skript can't understand this loop: '" + n.getName() + "'");
						continue;
					}
					currentLoops.add(new Pair<String, LoopVar<?>>(Skript.getLoopVarName((Class<? extends LoopVar<?>>) loopvar.getClass()), loopvar));
					items.add(new Loop(loopvar, loadItems((GroupNode) n)));
				} else if (n.getName().startsWith("if ")) {
					Skript.warning("'if' has not yet been implemented!");
				} else {
					Skript.error("invalid group, it should either start with 'loop' or 'if'.");
				}
			}
		}
		SkriptLogger.setNode(node);
		return items;
	}
	
	static int load(final Config config) {
		int i = 0;
		for (final ConfigNode node : config.getMainNode()) {
			if (node.isVoid())
				continue;
			if (!node.isGroup()) {
				Skript.error("triggers always have to have to start with an event");
				continue;
			}
			if (Skript.logVeryHigh())
				Skript.info("loading trigger '" + node.getName() + "'");
			final SkriptEvent event = SkriptEvent.parse(node.getName());
			if (event == null) {
				Skript.printWithheldError("Skript can't understand this event: '" + node.getName() + "'");
				continue;
			}
			final Class<? extends Event>[] events = Skript.getEvents(event.getClass());
			final Trigger trigger = new Trigger(node.getName(), event, loadItems((GroupNode) node));
			EventHandler.addTrigger(events, trigger);
			i++;
		}
		if (Skript.logHigh())
			Skript.info("loaded " + i + " trigger" + (i == 1 ? "" : "s") + (Skript.logVeryHigh() ? "" : " from '" + config.getFileName() + "'"));
		return i;
	}
	
	public static String[] split(final String s) {
		int c = -1, p = -1;
		final ArrayList<String> r = new ArrayList<String>();
		while (c != s.length()) {
			final int oldc = c;
			p = s.indexOf('%', c + 1);
			if (p == -1)
				p = s.length();
			c = s.indexOf(',', c + 1);
			if (c == -1)
				c = s.length();
			if (p < c) {
				p = s.indexOf('%', p + 1);
				if (c < p) {
					c = s.indexOf(',', p + 1);
					if (c == -1)
						c = s.length();
				}
			}
			r.add(s.substring(oldc + 1, c).trim());
		}
		return r.toArray(new String[r.size()]);
	}
	
}
