/*
 * Decompiled with CFR 0.152.
 */
package com.sedmelluq.discord.lavaplayer.remote;

import com.sedmelluq.discord.lavaplayer.player.DefaultAudioPlayerManager;
import com.sedmelluq.discord.lavaplayer.remote.RemoteAudioTrackExecutor;
import com.sedmelluq.discord.lavaplayer.remote.message.NodeStatisticsMessage;
import com.sedmelluq.discord.lavaplayer.remote.message.RemoteMessage;
import com.sedmelluq.discord.lavaplayer.remote.message.RemoteMessageMapper;
import com.sedmelluq.discord.lavaplayer.remote.message.TrackExceptionMessage;
import com.sedmelluq.discord.lavaplayer.remote.message.TrackFrameDataMessage;
import com.sedmelluq.discord.lavaplayer.remote.message.TrackFrameRequestMessage;
import com.sedmelluq.discord.lavaplayer.remote.message.TrackStartRequestMessage;
import com.sedmelluq.discord.lavaplayer.remote.message.TrackStartResponseMessage;
import com.sedmelluq.discord.lavaplayer.remote.message.TrackStoppedMessage;
import com.sedmelluq.discord.lavaplayer.tools.ExceptionTools;
import com.sedmelluq.discord.lavaplayer.tools.FriendlyException;
import com.sedmelluq.discord.lavaplayer.track.AudioTrack;
import com.sedmelluq.discord.lavaplayer.track.playback.AudioFrame;
import com.sedmelluq.discord.lavaplayer.track.playback.AudioFrameBuffer;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RemoteNodeProcessor
implements Runnable {
    private static final Logger log = LoggerFactory.getLogger(RemoteNodeProcessor.class);
    private static final int CONNECT_TIMEOUT = 1000;
    private static final int SOCKET_TIMEOUT = 2000;
    private static final int TRACK_KILL_THRESHOLD = 5000;
    private final DefaultAudioPlayerManager playerManager;
    private final String nodeAddress;
    private final ScheduledThreadPoolExecutor scheduledExecutor;
    private final BlockingQueue<RemoteMessage> queuedMessages;
    private final ConcurrentMap<Long, RemoteAudioTrackExecutor> playingTracks;
    private final RemoteMessageMapper mapper;
    private final HttpClientBuilder httpClientBuilder;
    private final AtomicBoolean threadRunning;
    private final AtomicInteger controlState;
    private volatile int aliveTickCounter;
    private volatile long lastAliveTime;
    private volatile NodeStatisticsMessage lastStatistics;

    public RemoteNodeProcessor(DefaultAudioPlayerManager playerManager, String nodeAddress, ScheduledThreadPoolExecutor scheduledExecutor) {
        this.playerManager = playerManager;
        this.nodeAddress = nodeAddress;
        this.scheduledExecutor = scheduledExecutor;
        this.queuedMessages = new LinkedBlockingQueue<RemoteMessage>();
        this.playingTracks = new ConcurrentHashMap<Long, RemoteAudioTrackExecutor>();
        this.mapper = new RemoteMessageMapper();
        this.httpClientBuilder = RemoteNodeProcessor.createHttpClientBuilder();
        this.threadRunning = new AtomicBoolean();
        this.controlState = new AtomicInteger(ControlState.OFFLINE.id());
    }

    public void startPlaying(RemoteAudioTrackExecutor executor) {
        AudioTrack track = executor.getTrack();
        if (this.playingTracks.putIfAbsent(executor.getExecutorId(), executor) == null) {
            log.info("Sending request to play {} {}", (Object)track.getIdentifier(), (Object)executor.getExecutorId());
            this.queuedMessages.add(new TrackStartRequestMessage(executor.getExecutorId(), track.getInfo(), this.playerManager.encodeTrackDetails(track), executor.getVolume(), executor.getConfiguration()));
        }
    }

    public void trackEnded(RemoteAudioTrackExecutor executor, boolean notifyNode) {
        if (this.playingTracks.remove(executor.getExecutorId()) != null) {
            log.info("Track {} removed from node {} (context {})", executor.getTrack().getIdentifier(), this.nodeAddress, executor.getExecutorId());
            if (notifyNode) {
                log.info("Notifying node {} of track stop for {} (context {})", this.nodeAddress, executor.getTrack().getIdentifier(), executor.getExecutorId());
                this.queuedMessages.add(new TrackStoppedMessage(executor.getExecutorId()));
            }
            executor.detach();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        if (!this.threadRunning.compareAndSet(false, true)) {
            log.debug("Not running node processor for {}, thread already active.", (Object)this.nodeAddress);
            return;
        }
        log.debug("Trying to connect to node {}.", (Object)this.nodeAddress);
        this.controlState.set(ControlState.PENDING.id());
        try (CloseableHttpClient httpClient = this.httpClientBuilder.build();){
            while (this.processOneTick(httpClient)) {
                this.aliveTickCounter = Math.max(1, this.aliveTickCounter + 1);
                this.lastAliveTime = System.currentTimeMillis();
            }
        }
        catch (Throwable e) {
            if (this.aliveTickCounter > 0) {
                log.error("Node {} went offline with exception.", (Object)this.nodeAddress, (Object)e);
            } else {
                log.debug("Retry, node {} is still offline.", (Object)this.nodeAddress);
            }
            ExceptionTools.rethrowErrors(e);
        }
        finally {
            this.controlState.set(ControlState.OFFLINE.id());
            this.aliveTickCounter = Math.min(-1, this.aliveTickCounter - 1);
            this.threadRunning.set(false);
            this.scheduledExecutor.schedule(this, this.getScheduleDelay(), TimeUnit.MILLISECONDS);
        }
    }

    private long getScheduleDelay() {
        if (this.aliveTickCounter >= -5) {
            return 1000L;
        }
        if (this.aliveTickCounter >= -20) {
            return 3000L;
        }
        return 10000L;
    }

    private boolean processOneTick(CloseableHttpClient httpClient) throws Exception {
        HttpPost post = new HttpPost("http://" + this.nodeAddress + "/tick");
        post.setEntity(new ByteArrayEntity(this.buildRequestBody()));
        try (CloseableHttpResponse response = httpClient.execute(post);){
            if (response.getStatusLine().getStatusCode() != 200) {
                throw new IOException("Returned an unexpected response code " + response.getStatusLine().getStatusCode());
            }
            if (this.controlState.compareAndSet(ControlState.PENDING.id(), ControlState.ONLINE.id())) {
                log.info("Node {} came online.", (Object)this.nodeAddress);
            } else if (this.controlState.get() != ControlState.ONLINE.id()) {
                log.warn("Node {} received successful response, but had already lost control of its tracks.");
                boolean bl = false;
                return bl;
            }
            this.lastAliveTime = System.currentTimeMillis();
            if (!this.handleResponseBody(response.getEntity().getContent())) {
                boolean bl = false;
                return bl;
            }
        }
        Thread.sleep(500L);
        return true;
    }

    private byte[] buildRequestBody() throws IOException {
        ByteArrayOutputStream outputBytes = new ByteArrayOutputStream();
        DataOutputStream output = new DataOutputStream(outputBytes);
        ArrayList<TrackFrameRequestMessage> messages = new ArrayList<TrackFrameRequestMessage>();
        int queuedCount = this.queuedMessages.drainTo(messages);
        if (queuedCount > 0) {
            log.debug("Including {} queued messages in the request to {}.", (Object)queuedCount, (Object)this.nodeAddress);
        }
        for (RemoteAudioTrackExecutor remoteAudioTrackExecutor : this.playingTracks.values()) {
            long pendingSeek = remoteAudioTrackExecutor.getPendingSeek();
            AudioFrameBuffer buffer = remoteAudioTrackExecutor.getAudioBuffer();
            int neededFrames = pendingSeek == -1L ? buffer.getRemainingCapacity() : buffer.getFullCapacity();
            messages.add(new TrackFrameRequestMessage(remoteAudioTrackExecutor.getExecutorId(), neededFrames, remoteAudioTrackExecutor.getVolume(), pendingSeek));
        }
        for (RemoteMessage remoteMessage : messages) {
            this.mapper.encode(output, remoteMessage);
        }
        this.mapper.endOutput(output);
        return outputBytes.toByteArray();
    }

    private boolean handleResponseBody(InputStream inputStream) {
        DataInputStream input = new DataInputStream(inputStream);
        try {
            RemoteMessage message;
            while ((message = this.mapper.decode(input)) != null) {
                if (message instanceof TrackStartResponseMessage) {
                    this.handleTrackStartResponse((TrackStartResponseMessage)message);
                    continue;
                }
                if (message instanceof TrackFrameDataMessage) {
                    this.handleTrackFrameData((TrackFrameDataMessage)message);
                    continue;
                }
                if (message instanceof TrackExceptionMessage) {
                    this.handleTrackException((TrackExceptionMessage)message);
                    continue;
                }
                if (!(message instanceof NodeStatisticsMessage)) continue;
                this.handleNodeStatistics((NodeStatisticsMessage)message);
            }
        }
        catch (InterruptedException interruption) {
            log.error("Node processing thread was interrupted.");
            Thread.currentThread().interrupt();
            return false;
        }
        catch (Throwable e) {
            log.error("Error when processing response from node.", e);
            ExceptionTools.rethrowErrors(e);
        }
        return true;
    }

    private void handleTrackStartResponse(TrackStartResponseMessage message) {
        if (message.success) {
            log.debug("Successful start confirmation from node {} for executor {}.", (Object)this.nodeAddress, (Object)message.executorId);
        } else {
            RemoteAudioTrackExecutor executor = (RemoteAudioTrackExecutor)this.playingTracks.get(message.executorId);
            if (executor != null) {
                executor.dispatchException(new FriendlyException("Remote machine failed to start track: " + message.failureReason, FriendlyException.Severity.SUSPICIOUS, null));
                executor.stop();
            } else {
                log.debug("Received failed track start for an already stopped executor {}.", (Object)message.executorId);
            }
        }
    }

    private void handleTrackFrameData(TrackFrameDataMessage message) throws Exception {
        RemoteAudioTrackExecutor executor = (RemoteAudioTrackExecutor)this.playingTracks.get(message.executorId);
        if (executor != null) {
            if (message.seekedPosition >= 0L) {
                executor.clearSeek(message.seekedPosition);
            }
            AudioFrameBuffer buffer = executor.getAudioBuffer();
            executor.receivedData();
            for (AudioFrame frame : message.frames) {
                buffer.consume(frame);
            }
            if (message.finished) {
                buffer.setTerminateOnEmpty();
                this.trackEnded(executor, false);
            }
        }
    }

    private void handleTrackException(TrackExceptionMessage message) {
        RemoteAudioTrackExecutor executor = (RemoteAudioTrackExecutor)this.playingTracks.get(message.executorId);
        if (executor != null) {
            executor.dispatchException(message.exception);
        }
    }

    private void handleNodeStatistics(NodeStatisticsMessage message) {
        log.trace("Received stats from node: {} {} {} {}", message.playingTrackCount, message.totalTrackCount, Float.valueOf(message.processCpuUsage), Float.valueOf(message.systemCpuUsage));
        this.lastStatistics = message;
    }

    private static HttpClientBuilder createHttpClientBuilder() {
        RequestConfig.Builder requestBuilder = RequestConfig.custom();
        requestBuilder = requestBuilder.setConnectTimeout(1000);
        requestBuilder = requestBuilder.setConnectionRequestTimeout(2000);
        HttpClientBuilder builder = HttpClientBuilder.create();
        builder.setDefaultRequestConfig(requestBuilder.build());
        return builder;
    }

    public synchronized void processHealthCheck(boolean terminate) {
        if (this.playingTracks.isEmpty() || !terminate && this.lastAliveTime >= System.currentTimeMillis() - 5000L) {
            return;
        }
        this.controlState.set(ControlState.OFFLINE.id());
        for (Long executorId : new ArrayList(this.playingTracks.keySet())) {
            RemoteAudioTrackExecutor executor = (RemoteAudioTrackExecutor)this.playingTracks.remove(executorId);
            if (executor == null) continue;
            executor.dispatchException(new FriendlyException("The machine processing this song went offline.", FriendlyException.Severity.SUSPICIOUS, null));
            executor.stop();
        }
    }

    public int getBalancerPenalty() {
        NodeStatisticsMessage statistics = this.lastStatistics;
        if (statistics == null || this.controlState.get() != ControlState.ONLINE.id()) {
            return Integer.MAX_VALUE;
        }
        int trackPenalty = statistics.totalTrackCount + statistics.playingTrackCount;
        int cpuPenalty = (int)Math.pow(statistics.systemCpuUsage + 0.7f, 10.0);
        return trackPenalty + cpuPenalty;
    }

    private static enum ControlState {
        PENDING,
        ONLINE,
        OFFLINE;


        private int id() {
            return this.ordinal();
        }
    }
}

