/*
 * Decompiled with CFR 0.152.
 */
package redis.clients.jedis.tests;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import redis.clients.jedis.HostAndPort;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.Pipeline;
import redis.clients.jedis.Response;
import redis.clients.jedis.Tuple;
import redis.clients.jedis.exceptions.JedisDataException;
import redis.clients.jedis.tests.HostAndPortUtil;
import redis.clients.util.SafeEncoder;

public class PipeliningTest
extends Assert {
    private static HostAndPort hnp = HostAndPortUtil.getRedisServers().get(0);
    private Jedis jedis;

    @Before
    public void setUp() throws Exception {
        this.jedis = new Jedis(hnp.getHost(), hnp.getPort(), 2000);
        this.jedis.connect();
        this.jedis.auth("foobared");
        this.jedis.flushAll();
    }

    @Test
    public void pipeline() throws UnsupportedEncodingException {
        Pipeline p = this.jedis.pipelined();
        p.set("foo", "bar");
        p.get("foo");
        List<Object> results = p.syncAndReturnAll();
        PipeliningTest.assertEquals((long)2L, (long)results.size());
        PipeliningTest.assertEquals((Object)"OK", (Object)results.get(0));
        PipeliningTest.assertEquals((Object)"bar", (Object)results.get(1));
    }

    @Test
    public void pipelineResponse() {
        this.jedis.set("string", "foo");
        this.jedis.lpush("list", "foo");
        this.jedis.hset("hash", "foo", "bar");
        this.jedis.zadd("zset", 1.0, "foo");
        this.jedis.sadd("set", "foo");
        Pipeline p = this.jedis.pipelined();
        Response<String> string = p.get("string");
        Response<String> list = p.lpop("list");
        Response<String> hash = p.hget("hash", "foo");
        Response<Set<String>> zset = p.zrange("zset", 0L, -1L);
        Response<String> set = p.spop("set");
        Response<Boolean> blist = p.exists("list");
        Response<Double> zincrby = p.zincrby("zset", 1.0, "foo");
        Response<Long> zcard = p.zcard("zset");
        p.lpush("list", "bar");
        Response<List<String>> lrange = p.lrange("list", 0L, -1L);
        Response<Map<String, String>> hgetAll = p.hgetAll("hash");
        p.sadd("set", "foo");
        Response<Set<String>> smembers = p.smembers("set");
        Response<Set<Tuple>> zrangeWithScores = p.zrangeWithScores("zset", 0L, -1L);
        p.sync();
        PipeliningTest.assertEquals((Object)"foo", (Object)string.get());
        PipeliningTest.assertEquals((Object)"foo", (Object)list.get());
        PipeliningTest.assertEquals((Object)"bar", (Object)hash.get());
        PipeliningTest.assertEquals((Object)"foo", (Object)zset.get().iterator().next());
        PipeliningTest.assertEquals((Object)"foo", (Object)set.get());
        PipeliningTest.assertEquals((Object)false, (Object)blist.get());
        PipeliningTest.assertEquals((Object)2.0, (Object)zincrby.get());
        PipeliningTest.assertEquals((Object)1L, (Object)zcard.get());
        PipeliningTest.assertEquals((long)1L, (long)lrange.get().size());
        PipeliningTest.assertNotNull((Object)hgetAll.get().get("foo"));
        PipeliningTest.assertEquals((long)1L, (long)smembers.get().size());
        PipeliningTest.assertEquals((long)1L, (long)zrangeWithScores.get().size());
    }

    @Test
    public void pipelineResponseWithData() {
        this.jedis.zadd("zset", 1.0, "foo");
        Pipeline p = this.jedis.pipelined();
        Response<Double> score = p.zscore("zset", "foo");
        p.sync();
        PipeliningTest.assertNotNull((Object)score.get());
    }

    @Test
    public void pipelineBinarySafeHashCommands() {
        this.jedis.hset("key".getBytes(), "f1".getBytes(), "v111".getBytes());
        this.jedis.hset("key".getBytes(), "f22".getBytes(), "v2222".getBytes());
        Pipeline p = this.jedis.pipelined();
        Response<Map<byte[], byte[]>> fmap = p.hgetAll("key".getBytes());
        Response<Set<byte[]>> fkeys = p.hkeys("key".getBytes());
        Response<List<byte[]>> fordered = p.hmget("key".getBytes(), "f22".getBytes(), "f1".getBytes());
        Response<List<byte[]>> fvals = p.hvals("key".getBytes());
        p.sync();
        PipeliningTest.assertNotNull(fmap.get());
        Map<byte[], byte[]> map = fmap.get();
        Set<byte[]> mapKeys = map.keySet();
        Iterator<byte[]> iterMap = mapKeys.iterator();
        byte[] firstMapKey = iterMap.next();
        byte[] secondMapKey = iterMap.next();
        PipeliningTest.assertFalse((boolean)iterMap.hasNext());
        this.verifyHasBothValues(firstMapKey, secondMapKey, "f1".getBytes(), "f22".getBytes());
        byte[] firstMapValue = map.get(firstMapKey);
        byte[] secondMapValue = map.get(secondMapKey);
        this.verifyHasBothValues(firstMapValue, secondMapValue, "v111".getBytes(), "v2222".getBytes());
        PipeliningTest.assertNotNull(fkeys.get());
        Iterator<byte[]> iter = fkeys.get().iterator();
        byte[] firstKey = iter.next();
        byte[] secondKey = iter.next();
        PipeliningTest.assertFalse((boolean)iter.hasNext());
        this.verifyHasBothValues(firstKey, secondKey, "f1".getBytes(), "f22".getBytes());
        PipeliningTest.assertNotNull(fordered.get());
        PipeliningTest.assertArrayEquals((byte[])"v2222".getBytes(), (byte[])fordered.get().get(0));
        PipeliningTest.assertArrayEquals((byte[])"v111".getBytes(), (byte[])fordered.get().get(1));
        PipeliningTest.assertNotNull(fvals.get());
        PipeliningTest.assertEquals((long)2L, (long)fvals.get().size());
        byte[] firstValue = fvals.get().get(0);
        byte[] secondValue = fvals.get().get(1);
        this.verifyHasBothValues(firstValue, secondValue, "v111".getBytes(), "v2222".getBytes());
    }

    private void verifyHasBothValues(byte[] firstKey, byte[] secondKey, byte[] value1, byte[] value2) {
        PipeliningTest.assertFalse((boolean)Arrays.equals(firstKey, secondKey));
        PipeliningTest.assertTrue((Arrays.equals(firstKey, value1) || Arrays.equals(firstKey, value2) ? 1 : 0) != 0);
        PipeliningTest.assertTrue((Arrays.equals(secondKey, value1) || Arrays.equals(secondKey, value2) ? 1 : 0) != 0);
    }

    @Test
    public void pipelineSelect() {
        Pipeline p = this.jedis.pipelined();
        p.select(1);
        p.sync();
    }

    @Test
    public void pipelineResponseWithoutData() {
        this.jedis.zadd("zset", 1.0, "foo");
        Pipeline p = this.jedis.pipelined();
        Response<Double> score = p.zscore("zset", "bar");
        p.sync();
        PipeliningTest.assertNull((Object)score.get());
    }

    @Test(expected=JedisDataException.class)
    public void pipelineResponseWithinPipeline() {
        this.jedis.set("string", "foo");
        Pipeline p = this.jedis.pipelined();
        Response<String> string = p.get("string");
        string.get();
        p.sync();
    }

    @Test
    public void pipelineWithPubSub() {
        Pipeline pipelined = this.jedis.pipelined();
        Response<Long> p1 = pipelined.publish("foo", "bar");
        Response<Long> p2 = pipelined.publish("foo".getBytes(), "bar".getBytes());
        pipelined.sync();
        PipeliningTest.assertEquals((long)0L, (long)p1.get());
        PipeliningTest.assertEquals((long)0L, (long)p2.get());
    }

    @Test
    public void canRetrieveUnsetKey() {
        Pipeline p = this.jedis.pipelined();
        Response<String> shouldNotExist = p.get(UUID.randomUUID().toString());
        p.sync();
        PipeliningTest.assertNull((Object)shouldNotExist.get());
    }

    @Test
    public void piplineWithError() {
        Pipeline p = this.jedis.pipelined();
        p.set("foo", "bar");
        Response<Set<String>> error = p.smembers("foo");
        Response<String> r = p.get("foo");
        p.sync();
        try {
            error.get();
            PipeliningTest.fail();
        }
        catch (JedisDataException jedisDataException) {
            // empty catch block
        }
        PipeliningTest.assertEquals((Object)r.get(), (Object)"bar");
    }

    @Test
    public void multi() {
        Pipeline p = this.jedis.pipelined();
        p.multi();
        Response<Long> r1 = p.hincrBy("a", "f1", -1L);
        Response<Long> r2 = p.hincrBy("a", "f1", -2L);
        Response<List<Object>> r3 = p.exec();
        List<Object> result = p.syncAndReturnAll();
        PipeliningTest.assertEquals((Object)new Long(-1L), (Object)r1.get());
        PipeliningTest.assertEquals((Object)new Long(-3L), (Object)r2.get());
        PipeliningTest.assertEquals((long)4L, (long)result.size());
        PipeliningTest.assertEquals((Object)"OK", (Object)result.get(0));
        PipeliningTest.assertEquals((Object)"QUEUED", (Object)result.get(1));
        PipeliningTest.assertEquals((Object)"QUEUED", (Object)result.get(2));
        List multiResult = (List)result.get(3);
        PipeliningTest.assertEquals((Object)new Long(-1L), multiResult.get(0));
        PipeliningTest.assertEquals((Object)new Long(-3L), multiResult.get(1));
        PipeliningTest.assertEquals((Object)new Long(-1L), (Object)r3.get().get(0));
        PipeliningTest.assertEquals((Object)new Long(-3L), (Object)r3.get().get(1));
    }

    @Test
    public void multiWithMassiveRequests() {
        Pipeline p = this.jedis.pipelined();
        p.multi();
        ArrayList<Response<Boolean>> responseList = new ArrayList<Response<Boolean>>();
        int i = 0;
        while (i < 100000) {
            responseList.add(p.setbit("test", 1L, true));
            ++i;
        }
        Response<List<Object>> exec = p.exec();
        p.sync();
        exec.get();
        for (Response response : responseList) {
            response.get();
        }
    }

    @Test
    public void multiWithSync() {
        this.jedis.set("foo", "314");
        this.jedis.set("bar", "foo");
        this.jedis.set("hello", "world");
        Pipeline p = this.jedis.pipelined();
        Response<String> r1 = p.get("bar");
        p.multi();
        Response<String> r2 = p.get("foo");
        p.exec();
        Response<String> r3 = p.get("hello");
        p.sync();
        PipeliningTest.assertEquals((Object)"foo", (Object)r1.get());
        PipeliningTest.assertEquals((Object)"314", (Object)r2.get());
        PipeliningTest.assertEquals((Object)"world", (Object)r3.get());
    }

    @Test(expected=JedisDataException.class)
    public void pipelineExecShoudThrowJedisDataExceptionWhenNotInMulti() {
        Pipeline pipeline = this.jedis.pipelined();
        pipeline.exec();
    }

    @Test(expected=JedisDataException.class)
    public void pipelineDiscardShoudThrowJedisDataExceptionWhenNotInMulti() {
        Pipeline pipeline = this.jedis.pipelined();
        pipeline.discard();
    }

    @Test(expected=JedisDataException.class)
    public void pipelineMultiShoudThrowJedisDataExceptionWhenAlreadyInMulti() {
        Pipeline pipeline = this.jedis.pipelined();
        pipeline.multi();
        pipeline.set("foo", "3");
        pipeline.multi();
    }

    @Test(expected=JedisDataException.class)
    public void testJedisThowExceptionWhenInPipeline() {
        Pipeline pipeline = this.jedis.pipelined();
        pipeline.set("foo", "3");
        this.jedis.get("somekey");
        PipeliningTest.fail((String)"Can't use jedis instance when in Pipeline");
    }

    @Test
    public void testReuseJedisWhenPipelineIsEmpty() {
        Pipeline pipeline = this.jedis.pipelined();
        pipeline.set("foo", "3");
        pipeline.sync();
        String result = this.jedis.get("foo");
        PipeliningTest.assertEquals((Object)result, (Object)"3");
    }

    @Test
    public void testResetStateWhenInPipeline() {
        Pipeline pipeline = this.jedis.pipelined();
        pipeline.set("foo", "3");
        this.jedis.resetState();
        String result = this.jedis.get("foo");
        PipeliningTest.assertEquals((Object)result, (Object)"3");
    }

    @Test
    public void testDiscardInPipeline() {
        Pipeline pipeline = this.jedis.pipelined();
        pipeline.multi();
        pipeline.set("foo", "bar");
        Response<String> discard = pipeline.discard();
        Response<String> get = pipeline.get("foo");
        pipeline.sync();
        discard.get();
        get.get();
    }

    @Test
    public void testEval() {
        String script = "return 'success!'";
        Pipeline p = this.jedis.pipelined();
        Response<String> result = p.eval(script);
        p.sync();
        PipeliningTest.assertEquals((Object)"success!", (Object)result.get());
    }

    @Test
    public void testEvalWithBinary() {
        String script = "return 'success!'";
        Pipeline p = this.jedis.pipelined();
        Response<Object> result = p.eval(SafeEncoder.encode(script));
        p.sync();
        PipeliningTest.assertArrayEquals((byte[])SafeEncoder.encode("success!"), (byte[])((byte[])result.get()));
    }

    @Test
    public void testEvalKeyAndArg() {
        String key = "test";
        String arg = "3";
        String script = "redis.call('INCRBY', KEYS[1], ARGV[1]) redis.call('INCRBY', KEYS[1], ARGV[1])";
        Pipeline p = this.jedis.pipelined();
        p.set(key, "0");
        Response<String> result0 = p.eval(script, Arrays.asList(key), Arrays.asList(arg));
        p.incr(key);
        Response<String> result1 = p.eval(script, Arrays.asList(key), Arrays.asList(arg));
        Response<String> result2 = p.get(key);
        p.sync();
        PipeliningTest.assertNull((Object)result0.get());
        PipeliningTest.assertNull((Object)result1.get());
        PipeliningTest.assertEquals((Object)"13", (Object)result2.get());
    }

    @Test
    public void testEvalKeyAndArgWithBinary() {
        byte[] bKey = SafeEncoder.encode("test");
        byte[] bArg = SafeEncoder.encode("3");
        byte[] bScript = SafeEncoder.encode("redis.call('INCRBY', KEYS[1], ARGV[1]) redis.call('INCRBY', KEYS[1], ARGV[1])");
        Pipeline bP = this.jedis.pipelined();
        bP.set(bKey, SafeEncoder.encode("0"));
        Response<Object> bResult0 = bP.eval(bScript, Arrays.asList(new byte[][]{bKey}), Arrays.asList(new byte[][]{bArg}));
        bP.incr(bKey);
        Response<Object> bResult1 = bP.eval(bScript, Arrays.asList(new byte[][]{bKey}), Arrays.asList(new byte[][]{bArg}));
        Response<byte[]> bResult2 = bP.get(bKey);
        bP.sync();
        PipeliningTest.assertNull((Object)bResult0.get());
        PipeliningTest.assertNull((Object)bResult1.get());
        PipeliningTest.assertArrayEquals((byte[])SafeEncoder.encode("13"), (byte[])bResult2.get());
    }

    @Test
    public void testEvalNestedListsWithBinary() {
        byte[] bScript = SafeEncoder.encode("return { {KEYS[1]} , {2} }");
        byte[] bKey = SafeEncoder.encode("key1");
        Pipeline p = this.jedis.pipelined();
        Response<Object> result = p.eval(bScript, 1, (byte[][])new byte[][]{bKey});
        p.sync();
        List results = (List)result.get();
        PipeliningTest.assertThat((Object)((List)results.get(0)), this.listWithItem(bKey));
        PipeliningTest.assertThat((Object)((List)results.get(1)), this.listWithItem(2L));
    }

    @Test
    public void testEvalsha() {
        String script = "return 'success!'";
        String sha1 = this.jedis.scriptLoad(script);
        PipeliningTest.assertTrue((boolean)this.jedis.scriptExists(sha1));
        Pipeline p = this.jedis.pipelined();
        Response<String> result = p.evalsha(sha1);
        p.sync();
        PipeliningTest.assertEquals((Object)"success!", (Object)result.get());
    }

    @Test
    public void testEvalshaKeyAndArgWithBinary() {
        byte[] bKey = SafeEncoder.encode("test");
        byte[] bArg = SafeEncoder.encode("3");
        String script = "redis.call('INCRBY', KEYS[1], ARGV[1]) redis.call('INCRBY', KEYS[1], ARGV[1])";
        byte[] bScript = SafeEncoder.encode(script);
        byte[] bSha1 = this.jedis.scriptLoad(bScript);
        PipeliningTest.assertTrue((this.jedis.scriptExists(bSha1) == 1L ? 1 : 0) != 0);
        Pipeline p = this.jedis.pipelined();
        p.set(bKey, SafeEncoder.encode("0"));
        Response<Object> result0 = p.evalsha(bSha1, Arrays.asList(new byte[][]{bKey}), Arrays.asList(new byte[][]{bArg}));
        p.incr(bKey);
        Response<Object> result1 = p.evalsha(bSha1, Arrays.asList(new byte[][]{bKey}), Arrays.asList(new byte[][]{bArg}));
        Response<byte[]> result2 = p.get(bKey);
        p.sync();
        PipeliningTest.assertNull((Object)result0.get());
        PipeliningTest.assertNull((Object)result1.get());
        PipeliningTest.assertArrayEquals((byte[])SafeEncoder.encode("13"), (byte[])result2.get());
    }

    private <T> Matcher<Iterable<? super T>> listWithItem(T expected) {
        return CoreMatchers.hasItem((Matcher)CoreMatchers.equalTo(expected));
    }

    @Test
    public void testEvalshaKeyAndArg() {
        String key = "test";
        String arg = "3";
        String script = "redis.call('INCRBY', KEYS[1], ARGV[1]) redis.call('INCRBY', KEYS[1], ARGV[1])";
        String sha1 = this.jedis.scriptLoad(script);
        PipeliningTest.assertTrue((boolean)this.jedis.scriptExists(sha1));
        Pipeline p = this.jedis.pipelined();
        p.set(key, "0");
        Response<String> result0 = p.evalsha(sha1, Arrays.asList(key), Arrays.asList(arg));
        p.incr(key);
        Response<String> result1 = p.evalsha(sha1, Arrays.asList(key), Arrays.asList(arg));
        Response<String> result2 = p.get(key);
        p.sync();
        PipeliningTest.assertNull((Object)result0.get());
        PipeliningTest.assertNull((Object)result1.get());
        PipeliningTest.assertEquals((Object)"13", (Object)result2.get());
    }

    @Test
    public void testPipelinedTransactionResponse() {
        String key1 = "key1";
        String val1 = "val1";
        String key2 = "key2";
        String val2 = "val2";
        String key3 = "key3";
        String field1 = "field1";
        String field2 = "field2";
        String field3 = "field3";
        String field4 = "field4";
        String value1 = "value1";
        String value2 = "value2";
        String value3 = "value3";
        String value4 = "value4";
        HashMap<String, String> hashMap = new HashMap<String, String>();
        hashMap.put(field1, value1);
        hashMap.put(field2, value2);
        String key4 = "key4";
        HashMap<String, String> hashMap1 = new HashMap<String, String>();
        hashMap1.put(field3, value3);
        hashMap1.put(field4, value4);
        this.jedis.set(key1, val1);
        this.jedis.set(key2, val2);
        this.jedis.hmset(key3, hashMap);
        this.jedis.hmset(key4, hashMap1);
        Pipeline pipeline = this.jedis.pipelined();
        pipeline.multi();
        pipeline.get(key1);
        pipeline.hgetAll(key2);
        pipeline.hgetAll(key3);
        pipeline.get(key4);
        Response<List<Object>> response = pipeline.exec();
        pipeline.sync();
        List<Object> result = response.get();
        PipeliningTest.assertEquals((long)4L, (long)result.size());
        PipeliningTest.assertEquals((Object)"val1", (Object)result.get(0));
        PipeliningTest.assertTrue((boolean)(result.get(1) instanceof JedisDataException));
        Map hashMapReceived = (Map)result.get(2);
        Iterator iterator = hashMapReceived.keySet().iterator();
        String mapKey1 = (String)iterator.next();
        String mapKey2 = (String)iterator.next();
        PipeliningTest.assertFalse((boolean)iterator.hasNext());
        this.verifyHasBothValues(mapKey1, mapKey2, field1, field2);
        String mapValue1 = (String)hashMapReceived.get(mapKey1);
        String mapValue2 = (String)hashMapReceived.get(mapKey2);
        this.verifyHasBothValues(mapValue1, mapValue2, value1, value2);
        PipeliningTest.assertTrue((boolean)(result.get(3) instanceof JedisDataException));
    }

    @Test
    public void testSyncWithNoCommandQueued() {
        Jedis jedis2 = new Jedis(hnp.getHost(), hnp.getPort(), 500);
        Pipeline pipeline = jedis2.pipelined();
        pipeline.sync();
        jedis2.close();
        jedis2 = new Jedis(hnp.getHost(), hnp.getPort(), 500);
        pipeline = jedis2.pipelined();
        List<Object> resp = pipeline.syncAndReturnAll();
        PipeliningTest.assertTrue((boolean)resp.isEmpty());
        jedis2.close();
    }

    @Test
    public void testCloseable() throws IOException {
        Jedis jedis2 = new Jedis(hnp.getHost(), hnp.getPort(), 500);
        jedis2.auth("foobared");
        Pipeline pipeline = jedis2.pipelined();
        Response<String> retFuture1 = pipeline.set("a", "1");
        Response<String> retFuture2 = pipeline.set("b", "2");
        pipeline.close();
        retFuture1.get();
        retFuture2.get();
    }

    @Test
    public void testCloseableWithMulti() throws IOException {
        Jedis jedis2 = new Jedis(hnp.getHost(), hnp.getPort(), 500);
        jedis2.auth("foobared");
        Pipeline pipeline = jedis2.pipelined();
        Response<String> retFuture1 = pipeline.set("a", "1");
        Response<String> retFuture2 = pipeline.set("b", "2");
        pipeline.multi();
        pipeline.set("a", "a");
        pipeline.set("b", "b");
        pipeline.close();
        try {
            pipeline.exec();
            PipeliningTest.fail((String)"close should discard transaction");
        }
        catch (JedisDataException e) {
            PipeliningTest.assertTrue((boolean)e.getMessage().contains("EXEC without MULTI"));
        }
        retFuture1.get();
        retFuture2.get();
    }

    private void verifyHasBothValues(String firstKey, String secondKey, String value1, String value2) {
        PipeliningTest.assertFalse((boolean)firstKey.equals(secondKey));
        PipeliningTest.assertTrue((firstKey.equals(value1) || firstKey.equals(value2) ? 1 : 0) != 0);
        PipeliningTest.assertTrue((secondKey.equals(value1) || secondKey.equals(value2) ? 1 : 0) != 0);
    }
}

