/*
 * Decompiled with CFR 0.152.
 */
package com.mongodb.internal.async.client.gridfs;

import com.mongodb.MongoGridFSException;
import com.mongodb.assertions.Assertions;
import com.mongodb.client.gridfs.model.GridFSFile;
import com.mongodb.diagnostics.logging.Logger;
import com.mongodb.diagnostics.logging.Loggers;
import com.mongodb.internal.async.AsyncBatchCursor;
import com.mongodb.internal.async.ErrorHandlingResultCallback;
import com.mongodb.internal.async.SingleResultCallback;
import com.mongodb.internal.async.client.AsyncClientSession;
import com.mongodb.internal.async.client.AsyncFindIterable;
import com.mongodb.internal.async.client.AsyncMongoCollection;
import com.mongodb.internal.async.client.gridfs.AsyncGridFSDownloadStream;
import com.mongodb.internal.async.client.gridfs.AsyncGridFSFindIterable;
import com.mongodb.lang.Nullable;
import java.nio.ByteBuffer;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedQueue;
import org.bson.Document;
import org.bson.types.Binary;

final class AsyncGridFSDownloadStreamImpl
implements AsyncGridFSDownloadStream {
    private static final Logger LOGGER = Loggers.getLogger("client.gridfs");
    private final AsyncClientSession clientSession;
    private final AsyncGridFSFindIterable fileInfoIterable;
    private final AsyncMongoCollection<Document> chunksCollection;
    private final ConcurrentLinkedQueue<Document> resultsQueue = new ConcurrentLinkedQueue();
    private final Object closeAndReadingLock = new Object();
    private boolean reading;
    private boolean closed;
    private GridFSFile fileInfo;
    private int numberOfChunks;
    private AsyncBatchCursor<Document> cursor;
    private int batchSize;
    private int chunkIndex;
    private int bufferOffset;
    private long currentPosition;
    private byte[] buffer = null;

    AsyncGridFSDownloadStreamImpl(@Nullable AsyncClientSession clientSession, AsyncGridFSFindIterable fileInfoIterable, AsyncMongoCollection<Document> chunksCollection) {
        this.clientSession = clientSession;
        this.fileInfoIterable = Assertions.notNull("file information", fileInfoIterable);
        this.chunksCollection = Assertions.notNull("chunks collection", chunksCollection);
    }

    @Override
    public void getGridFSFile(SingleResultCallback<GridFSFile> callback) {
        Assertions.notNull("callback", callback);
        final SingleResultCallback<GridFSFile> errHandlingCallback = ErrorHandlingResultCallback.errorHandlingCallback(callback, LOGGER);
        if (this.hasFileInfo()) {
            errHandlingCallback.onResult(this.fileInfo, null);
            return;
        }
        if (!this.tryGetReadingLock(errHandlingCallback)) {
            return;
        }
        this.fileInfoIterable.first(new SingleResultCallback<GridFSFile>(){

            @Override
            public void onResult(GridFSFile result, Throwable t) {
                AsyncGridFSDownloadStreamImpl.this.releaseReadingLock();
                if (t != null) {
                    errHandlingCallback.onResult(null, t);
                } else if (result == null) {
                    errHandlingCallback.onResult(null, new MongoGridFSException("File not found"));
                } else {
                    AsyncGridFSDownloadStreamImpl.this.fileInfo = result;
                    AsyncGridFSDownloadStreamImpl.this.numberOfChunks = (int)Math.ceil((double)AsyncGridFSDownloadStreamImpl.this.fileInfo.getLength() / (double)AsyncGridFSDownloadStreamImpl.this.fileInfo.getChunkSize());
                    errHandlingCallback.onResult(result, null);
                }
            }
        });
    }

    @Override
    public AsyncGridFSDownloadStream batchSize(int batchSize) {
        Assertions.isTrueArgument("batchSize cannot be negative", batchSize >= 0);
        this.batchSize = batchSize;
        this.discardCursor();
        return this;
    }

    @Override
    public void read(final ByteBuffer dst, SingleResultCallback<Integer> callback) {
        Assertions.notNull("dst", dst);
        Assertions.notNull("callback", callback);
        final SingleResultCallback<Integer> errHandlingCallback = ErrorHandlingResultCallback.errorHandlingCallback(callback, LOGGER);
        if (!this.hasFileInfo()) {
            this.getGridFSFile(new SingleResultCallback<GridFSFile>(){

                @Override
                public void onResult(GridFSFile result, Throwable t) {
                    if (t != null) {
                        errHandlingCallback.onResult(null, t);
                    } else {
                        AsyncGridFSDownloadStreamImpl.this.read(dst, errHandlingCallback);
                    }
                }
            });
            return;
        }
        if (!this.tryGetReadingLock(errHandlingCallback)) {
            return;
        }
        if (this.currentPosition == this.fileInfo.getLength()) {
            this.releaseReadingLock();
            errHandlingCallback.onResult(-1, null);
            return;
        }
        this.checkAndFetchResults(0, dst, new SingleResultCallback<Integer>(){

            @Override
            public void onResult(Integer result, Throwable t) {
                AsyncGridFSDownloadStreamImpl.this.releaseReadingLock();
                errHandlingCallback.onResult(result, t);
            }
        });
    }

    @Override
    public void skip(final long bytesToSkip, SingleResultCallback<Long> callback) {
        Assertions.notNull("callback", callback);
        final SingleResultCallback<Long> errHandlingCallback = ErrorHandlingResultCallback.errorHandlingCallback(callback, LOGGER);
        if (this.checkClosed()) {
            errHandlingCallback.onResult(null, null);
        } else if (!this.hasFileInfo()) {
            this.getGridFSFile(new SingleResultCallback<GridFSFile>(){

                @Override
                public void onResult(GridFSFile result, Throwable t) {
                    if (t != null) {
                        errHandlingCallback.onResult(null, t);
                    } else {
                        AsyncGridFSDownloadStreamImpl.this.skip(bytesToSkip, errHandlingCallback);
                    }
                }
            });
        } else if (bytesToSkip <= 0L) {
            callback.onResult(0L, null);
        } else {
            long skippedPosition = this.currentPosition + bytesToSkip;
            this.bufferOffset = (int)(skippedPosition % (long)this.fileInfo.getChunkSize());
            if (skippedPosition >= this.fileInfo.getLength()) {
                long skipped = this.fileInfo.getLength() - this.currentPosition;
                this.chunkIndex = this.numberOfChunks - 1;
                this.currentPosition = this.fileInfo.getLength();
                this.buffer = null;
                this.discardCursor();
                callback.onResult(skipped, null);
            } else {
                int newChunkIndex = (int)Math.floor((double)skippedPosition / (double)this.fileInfo.getChunkSize());
                if (this.chunkIndex != newChunkIndex) {
                    this.chunkIndex = newChunkIndex;
                    this.buffer = null;
                    this.discardCursor();
                }
                this.currentPosition += bytesToSkip;
                callback.onResult(bytesToSkip, null);
            }
        }
    }

    private void checkAndFetchResults(final int amountRead, final ByteBuffer dst, final SingleResultCallback<Integer> callback) {
        if (this.currentPosition == this.fileInfo.getLength() || dst.remaining() == 0) {
            callback.onResult(amountRead, null);
        } else if (this.hasResultsToProcess()) {
            this.processResults(amountRead, dst, callback);
        } else if (this.cursor == null) {
            Document filter = new Document("files_id", this.fileInfo.getId()).append("n", new Document("$gte", this.chunkIndex));
            AsyncFindIterable<Document> findIterable = this.clientSession != null ? this.chunksCollection.find(this.clientSession, filter) : this.chunksCollection.find(filter);
            findIterable.batchSize(this.batchSize).sort(new Document("n", 1)).batchCursor(new SingleResultCallback<AsyncBatchCursor<Document>>(){

                @Override
                public void onResult(AsyncBatchCursor<Document> result, Throwable t) {
                    if (t != null) {
                        callback.onResult(null, t);
                    } else {
                        AsyncGridFSDownloadStreamImpl.this.cursor = result;
                        AsyncGridFSDownloadStreamImpl.this.checkAndFetchResults(amountRead, dst, callback);
                    }
                }
            });
        } else {
            this.cursor.next(new SingleResultCallback<List<Document>>(){

                @Override
                public void onResult(List<Document> result, Throwable t) {
                    if (t != null) {
                        callback.onResult(null, t);
                    } else if (result == null || result.isEmpty()) {
                        callback.onResult(null, AsyncGridFSDownloadStreamImpl.this.chunkNotFound(AsyncGridFSDownloadStreamImpl.this.chunkIndex));
                    } else {
                        AsyncGridFSDownloadStreamImpl.this.resultsQueue.addAll(result);
                        if (AsyncGridFSDownloadStreamImpl.this.batchSize == 1) {
                            AsyncGridFSDownloadStreamImpl.this.discardCursor();
                        }
                        AsyncGridFSDownloadStreamImpl.this.processResults(amountRead, dst, callback);
                    }
                }
            });
        }
    }

    private void processResults(int previousAmountRead, ByteBuffer dst, SingleResultCallback<Integer> callback) {
        try {
            int amountRead = previousAmountRead;
            int amountToCopy = dst.remaining();
            while (this.currentPosition < this.fileInfo.getLength() && amountToCopy > 0) {
                if (this.getBufferFromResultsQueue()) {
                    boolean wasEndOfBuffer = this.buffer != null && this.bufferOffset == this.buffer.length;
                    this.buffer = this.getBufferFromChunk(this.resultsQueue.poll(), this.chunkIndex);
                    ++this.chunkIndex;
                    if (wasEndOfBuffer) {
                        this.bufferOffset = 0;
                    }
                }
                if (amountToCopy > this.buffer.length - this.bufferOffset) {
                    amountToCopy = this.buffer.length - this.bufferOffset;
                }
                if (amountToCopy <= 0) continue;
                dst.put(this.buffer, this.bufferOffset, amountToCopy);
                this.bufferOffset += amountToCopy;
                this.currentPosition += (long)amountToCopy;
                amountRead += amountToCopy;
                amountToCopy = dst.remaining();
            }
            this.checkAndFetchResults(amountRead, dst, callback);
        }
        catch (MongoGridFSException e) {
            callback.onResult(null, e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close(SingleResultCallback<Void> callback) {
        Assertions.notNull("callback", callback);
        SingleResultCallback<Void> errHandlingCallback = ErrorHandlingResultCallback.errorHandlingCallback(callback, LOGGER);
        if (this.checkClosed()) {
            errHandlingCallback.onResult(null, null);
        } else if (!this.getReadingLock()) {
            this.callbackIsReadingException(callback);
        } else {
            Object object = this.closeAndReadingLock;
            synchronized (object) {
                if (!this.closed) {
                    this.closed = true;
                }
            }
            this.discardCursor();
            errHandlingCallback.onResult(null, null);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean hasFileInfo() {
        boolean hasInfo = false;
        Object object = this.closeAndReadingLock;
        synchronized (object) {
            hasInfo = this.fileInfo != null;
        }
        return hasInfo;
    }

    private MongoGridFSException chunkNotFound(int chunkIndex) {
        return new MongoGridFSException(String.format("Could not find file chunk for files_id: %s at chunk index %s.", this.fileInfo.getId(), chunkIndex));
    }

    private byte[] getBufferFromChunk(Document chunk, int expectedChunkIndex) {
        long expectedDataLength;
        if (chunk == null || chunk.getInteger("n") != expectedChunkIndex) {
            throw this.chunkNotFound(expectedChunkIndex);
        }
        if (!(chunk.get("data") instanceof Binary)) {
            throw new MongoGridFSException("Unexpected data format for the chunk");
        }
        byte[] data = ((Binary)((Object)chunk.get((Object)"data", Binary.class))).getData();
        long l = expectedDataLength = expectedChunkIndex + 1 == this.numberOfChunks ? this.fileInfo.getLength() - (long)expectedChunkIndex * (long)this.fileInfo.getChunkSize() : (long)this.fileInfo.getChunkSize();
        if ((long)data.length != expectedDataLength) {
            throw new MongoGridFSException(String.format("Chunk size data length is not the expected size. The size was %s for file_id: %s chunk index %s it should be %s bytes.", data.length, this.fileInfo.getId(), expectedChunkIndex, expectedDataLength));
        }
        return data;
    }

    private boolean getBufferFromResultsQueue() {
        return !this.resultsQueue.isEmpty() && (this.buffer == null || this.bufferOffset == this.buffer.length);
    }

    private boolean hasResultsToProcess() {
        return !this.resultsQueue.isEmpty() || this.buffer != null && this.bufferOffset < this.buffer.length;
    }

    private <A> boolean tryGetReadingLock(SingleResultCallback<A> callback) {
        if (this.checkClosed()) {
            this.callbackClosedException(callback);
            return false;
        }
        if (!this.getReadingLock()) {
            this.callbackIsReadingException(callback);
            return false;
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean checkClosed() {
        Object object = this.closeAndReadingLock;
        synchronized (object) {
            return this.closed;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean getReadingLock() {
        boolean gotLock = false;
        Object object = this.closeAndReadingLock;
        synchronized (object) {
            if (!this.reading) {
                this.reading = true;
                gotLock = true;
            }
        }
        return gotLock;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void releaseReadingLock() {
        Object object = this.closeAndReadingLock;
        synchronized (object) {
            this.reading = false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void discardCursor() {
        Object object = this.closeAndReadingLock;
        synchronized (object) {
            if (this.cursor != null) {
                this.cursor.close();
                this.cursor = null;
            }
        }
    }

    private <T> void callbackClosedException(SingleResultCallback<T> callback) {
        callback.onResult(null, new MongoGridFSException("The AsyncInputStream has been closed"));
    }

    private <T> void callbackIsReadingException(SingleResultCallback<T> callback) {
        callback.onResult(null, new MongoGridFSException("The AsyncInputStream does not support concurrent reading."));
    }
}

