/*
 * Decompiled with CFR 0.152.
 */
package com.zaxxer.hikari.pool;

import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.health.HealthCheckRegistry;
import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariPoolMXBean;
import com.zaxxer.hikari.metrics.MetricsTrackerFactory;
import com.zaxxer.hikari.metrics.PoolStats;
import com.zaxxer.hikari.metrics.dropwizard.CodahaleHealthChecker;
import com.zaxxer.hikari.metrics.dropwizard.CodahaleMetricsTrackerFactory;
import com.zaxxer.hikari.pool.PoolBase;
import com.zaxxer.hikari.pool.PoolEntry;
import com.zaxxer.hikari.pool.ProxyConnection;
import com.zaxxer.hikari.pool.ProxyLeakTask;
import com.zaxxer.hikari.util.ClockSource;
import com.zaxxer.hikari.util.ConcurrentBag;
import com.zaxxer.hikari.util.DefaultThreadFactory;
import com.zaxxer.hikari.util.SuspendResumeLock;
import com.zaxxer.hikari.util.UtilityElf;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.SQLTransientConnectionException;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HikariPool
extends PoolBase
implements HikariPoolMXBean,
ConcurrentBag.IBagStateListener {
    private static final Logger LOGGER = LoggerFactory.getLogger(HikariPool.class);
    private static final ClockSource clockSource = ClockSource.INSTANCE;
    private final long ALIVE_BYPASS_WINDOW_MS = Long.getLong("com.zaxxer.hikari.aliveBypassWindowMs", TimeUnit.MILLISECONDS.toMillis(500L));
    private final long HOUSEKEEPING_PERIOD_MS = Long.getLong("com.zaxxer.hikari.housekeeping.periodMs", TimeUnit.SECONDS.toMillis(30L));
    private final PoolEntryCreator POOL_ENTRY_CREATOR = new PoolEntryCreator();
    private static final int POOL_NORMAL = 0;
    private static final int POOL_SUSPENDED = 1;
    private static final int POOL_SHUTDOWN = 2;
    private volatile int poolState;
    private final AtomicInteger totalConnections;
    private final ThreadPoolExecutor addConnectionExecutor;
    private final ThreadPoolExecutor closeConnectionExecutor;
    private final ScheduledThreadPoolExecutor houseKeepingExecutorService;
    private final ConcurrentBag<PoolEntry> connectionBag = new ConcurrentBag(this);
    private final ProxyLeakTask leakTask;
    private final SuspendResumeLock suspendResumeLock;
    private PoolBase.MetricsTrackerDelegate metricsTracker;
    private boolean isRecordMetrics;

    public HikariPool(HikariConfig config) {
        super(config);
        this.totalConnections = new AtomicInteger();
        this.suspendResumeLock = config.isAllowPoolSuspension() ? new SuspendResumeLock() : SuspendResumeLock.FAUX_LOCK;
        this.addConnectionExecutor = UtilityElf.createThreadPoolExecutor(config.getMaximumPoolSize(), "Hikari connection adder (pool " + this.poolName + ")", config.getThreadFactory(), new ThreadPoolExecutor.DiscardPolicy());
        this.closeConnectionExecutor = UtilityElf.createThreadPoolExecutor(4, "Hikari connection closer (pool " + this.poolName + ")", config.getThreadFactory(), new ThreadPoolExecutor.CallerRunsPolicy());
        if (config.getScheduledExecutorService() == null) {
            ThreadFactory threadFactory = config.getThreadFactory() != null ? config.getThreadFactory() : new DefaultThreadFactory("Hikari housekeeper (pool " + this.poolName + ")", true);
            this.houseKeepingExecutorService = new ScheduledThreadPoolExecutor(1, threadFactory, new ThreadPoolExecutor.DiscardPolicy());
            this.houseKeepingExecutorService.setExecuteExistingDelayedTasksAfterShutdownPolicy(false);
            this.houseKeepingExecutorService.setRemoveOnCancelPolicy(true);
        } else {
            this.houseKeepingExecutorService = config.getScheduledExecutorService();
        }
        this.houseKeepingExecutorService.scheduleAtFixedRate(new HouseKeeper(), this.HOUSEKEEPING_PERIOD_MS, this.HOUSEKEEPING_PERIOD_MS, TimeUnit.MILLISECONDS);
        this.leakTask = new ProxyLeakTask(config.getLeakDetectionThreshold(), this.houseKeepingExecutorService);
        if (config.getMetricsTrackerFactory() != null) {
            this.setMetricsTrackerFactory(config.getMetricsTrackerFactory());
        } else {
            this.setMetricRegistry(config.getMetricRegistry());
        }
        this.setHealthCheckRegistry(config.getHealthCheckRegistry());
        this.registerMBeans(this);
        this.initializeConnections();
    }

    public final Connection getConnection() throws SQLException {
        return this.getConnection(this.connectionTimeout);
    }

    public final Connection getConnection(long hardTimeout) throws SQLException {
        this.suspendResumeLock.acquire();
        long startTime = clockSource.currentTime();
        try {
            long timeout = hardTimeout;
            do {
                PoolEntry poolEntry;
                if ((poolEntry = this.connectionBag.borrow(timeout, TimeUnit.MILLISECONDS)) == null) {
                    break;
                }
                long now = clockSource.currentTime();
                if (!poolEntry.isMarkedEvicted() && (clockSource.elapsedMillis(poolEntry.lastAccessed, now) <= this.ALIVE_BYPASS_WINDOW_MS || this.isConnectionAlive(poolEntry.connection))) {
                    this.metricsTracker.recordBorrowStats(poolEntry, startTime);
                    Connection connection = poolEntry.createProxyConnection(this.leakTask.start(poolEntry), now);
                    return connection;
                }
                this.closeConnection(poolEntry, "(connection evicted or dead)");
                timeout = hardTimeout - clockSource.elapsedMillis(startTime);
            } while (timeout > 0L);
        }
        catch (InterruptedException e) {
            throw new SQLException(this.poolName + " - Interrupted during connection acquisition", e);
        }
        finally {
            this.suspendResumeLock.release();
        }
        this.logPoolState("Timeout failure\t");
        String sqlState = null;
        Throwable originalException = this.getLastConnectionFailure();
        if (originalException instanceof SQLException) {
            sqlState = ((SQLException)originalException).getSQLState();
        }
        SQLTransientConnectionException connectionException = new SQLTransientConnectionException(this.poolName + " - Connection is not available, request timed out after " + clockSource.elapsedMillis(startTime) + "ms.", sqlState, originalException);
        if (originalException instanceof SQLException) {
            connectionException.setNextException((SQLException)originalException);
        }
        throw connectionException;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public final synchronized void shutdown() throws InterruptedException {
        try {
            this.poolState = 2;
            LOGGER.info("{} - is closing down.", (Object)this.poolName);
            this.logPoolState("Before closing\t");
            this.softEvictConnections();
            this.addConnectionExecutor.shutdown();
            this.addConnectionExecutor.awaitTermination(5L, TimeUnit.SECONDS);
            if (this.config.getScheduledExecutorService() == null) {
                this.houseKeepingExecutorService.shutdown();
                this.houseKeepingExecutorService.awaitTermination(5L, TimeUnit.SECONDS);
            }
            this.connectionBag.close();
            ThreadPoolExecutor assassinExecutor = UtilityElf.createThreadPoolExecutor(this.config.getMaximumPoolSize(), "Hikari connection assassin (pool " + this.poolName + ")", this.config.getThreadFactory(), new ThreadPoolExecutor.CallerRunsPolicy());
            try {
                long start = clockSource.currentTime();
                do {
                    this.abortActiveConnections(assassinExecutor);
                    this.softEvictConnections();
                } while (this.getTotalConnections() > 0 && clockSource.elapsedMillis(start) < TimeUnit.SECONDS.toMillis(5L));
            }
            finally {
                assassinExecutor.shutdown();
                assassinExecutor.awaitTermination(5L, TimeUnit.SECONDS);
            }
            this.shutdownNetworkTimeoutExecutor();
            this.closeConnectionExecutor.shutdown();
            this.closeConnectionExecutor.awaitTermination(5L, TimeUnit.SECONDS);
        }
        catch (Throwable throwable) {
            this.logPoolState("After closing\t");
            this.unregisterMBeans();
            this.metricsTracker.close();
            throw throwable;
        }
        this.logPoolState("After closing\t");
        this.unregisterMBeans();
        this.metricsTracker.close();
    }

    public final void evictConnection(Connection connection) {
        ProxyConnection proxyConnection = (ProxyConnection)connection;
        proxyConnection.cancelLeakTask();
        this.softEvictConnection(proxyConnection.getPoolEntry(), "(connection evicted by user)", true);
    }

    public void setMetricRegistry(Object metricRegistry) {
        boolean bl = this.isRecordMetrics = metricRegistry != null;
        if (this.isRecordMetrics) {
            this.setMetricsTrackerFactory(new CodahaleMetricsTrackerFactory((MetricRegistry)metricRegistry));
        } else {
            this.setMetricsTrackerFactory(null);
        }
    }

    public void setMetricsTrackerFactory(MetricsTrackerFactory metricsTrackerFactory) {
        this.isRecordMetrics = metricsTrackerFactory != null;
        this.metricsTracker = this.isRecordMetrics ? new PoolBase.MetricsTrackerDelegate(metricsTrackerFactory.create(this.config.getPoolName(), this.getPoolStats())) : new PoolBase.NopMetricsTrackerDelegate();
    }

    public void setHealthCheckRegistry(Object healthCheckRegistry) {
        if (healthCheckRegistry != null) {
            CodahaleHealthChecker.registerHealthChecks(this, this.config, (HealthCheckRegistry)healthCheckRegistry);
        }
    }

    @Override
    public Future<Boolean> addBagItem() {
        return this.addConnectionExecutor.submit(this.POOL_ENTRY_CREATOR);
    }

    @Override
    public final int getActiveConnections() {
        return this.connectionBag.getCount(1);
    }

    @Override
    public final int getIdleConnections() {
        return this.connectionBag.getCount(0);
    }

    @Override
    public final int getTotalConnections() {
        return this.connectionBag.size() - this.connectionBag.getCount(-1);
    }

    @Override
    public final int getThreadsAwaitingConnection() {
        return this.connectionBag.getPendingQueue();
    }

    @Override
    public void softEvictConnections() {
        for (PoolEntry poolEntry : this.connectionBag.values()) {
            this.softEvictConnection(poolEntry, "(connection evicted by user)", false);
        }
    }

    @Override
    public final synchronized void suspendPool() {
        if (this.suspendResumeLock == SuspendResumeLock.FAUX_LOCK) {
            throw new IllegalStateException(this.poolName + " - is not suspendable");
        }
        if (this.poolState != 1) {
            this.suspendResumeLock.suspend();
            this.poolState = 1;
        }
    }

    @Override
    public final synchronized void resumePool() {
        if (this.poolState == 1) {
            this.poolState = 0;
            this.fillPool();
            this.suspendResumeLock.resume();
        }
    }

    final void logPoolState(String ... prefix) {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("{}pool {} stats (total={}, active={}, idle={}, waiting={})", prefix.length > 0 ? prefix[0] : "", this.poolName, this.getTotalConnections(), this.getActiveConnections(), this.getIdleConnections(), this.getThreadsAwaitingConnection());
        }
    }

    @Override
    final void releaseConnection(PoolEntry poolEntry) {
        this.metricsTracker.recordConnectionUsage(poolEntry);
        this.connectionBag.requite(poolEntry);
    }

    final void closeConnection(PoolEntry poolEntry, final String closureReason) {
        if (this.connectionBag.remove(poolEntry)) {
            final Connection connection = poolEntry.connection;
            poolEntry.close();
            int tc = this.totalConnections.decrementAndGet();
            if (tc < 0) {
                LOGGER.warn("{} - Internal accounting inconsistency, totalConnections={}", this.poolName, tc, new Exception());
            }
            this.closeConnectionExecutor.execute(new Runnable(){

                @Override
                public void run() {
                    HikariPool.this.quietlyCloseConnection(connection, closureReason);
                }
            });
        }
    }

    private PoolEntry createPoolEntry() {
        try {
            final PoolEntry poolEntry = this.newPoolEntry();
            long maxLifetime = this.config.getMaxLifetime();
            if (maxLifetime > 0L) {
                long variance = maxLifetime > 10000L ? ThreadLocalRandom.current().nextLong(Math.max(10000L, maxLifetime / 40L)) : 0L;
                long lifetime = maxLifetime - variance;
                poolEntry.setFutureEol(this.houseKeepingExecutorService.schedule(new Runnable(){

                    @Override
                    public void run() {
                        HikariPool.this.softEvictConnection(poolEntry, "(connection reached maxLifetime)", false);
                    }
                }, lifetime, TimeUnit.MILLISECONDS));
            }
            LOGGER.debug("{} - Added connection {}", (Object)this.poolName, (Object)poolEntry.connection);
            return poolEntry;
        }
        catch (Exception e) {
            if (this.poolState == 0) {
                LOGGER.debug("{} - Cannot acquire connection from data source", (Object)this.poolName, (Object)e);
            }
            return null;
        }
    }

    private void fillPool() {
        int connectionsToAdd = Math.min(this.config.getMaximumPoolSize() - this.totalConnections.get(), this.config.getMinimumIdle() - this.getIdleConnections());
        for (int i = 0; i < connectionsToAdd; ++i) {
            this.addBagItem();
        }
        if (connectionsToAdd > 0 && LOGGER.isDebugEnabled()) {
            this.addConnectionExecutor.execute(new Runnable(){

                @Override
                public void run() {
                    HikariPool.this.logPoolState("After adding\t");
                }
            });
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void abortActiveConnections(ExecutorService assassinExecutor) {
        for (PoolEntry poolEntry : this.connectionBag.values(1)) {
            try {
                poolEntry.connection.abort(assassinExecutor);
            }
            catch (Throwable e) {
                this.quietlyCloseConnection(poolEntry.connection, "(connection aborted during shutdown)");
            }
            finally {
                poolEntry.close();
                if (!this.connectionBag.remove(poolEntry)) continue;
                this.totalConnections.decrementAndGet();
            }
        }
    }

    private void initializeConnections() {
        if (this.config.isInitializationFailFast()) {
            try {
                Connection connection = this.getConnection();
                if (this.config.getMinimumIdle() == 0) {
                    this.evictConnection(connection);
                } else {
                    connection.close();
                }
            }
            catch (Throwable e) {
                try {
                    this.shutdown();
                }
                catch (Throwable ex) {
                    e.addSuppressed(ex);
                }
                throw new PoolInitializationException(e);
            }
        }
        this.fillPool();
    }

    private void softEvictConnection(PoolEntry poolEntry, String reason, boolean owner) {
        if (owner || this.connectionBag.reserve(poolEntry)) {
            this.closeConnection(poolEntry, reason);
        } else {
            poolEntry.markEvicted();
        }
    }

    private PoolStats getPoolStats() {
        return new PoolStats(TimeUnit.SECONDS.toMillis(1L)){

            @Override
            protected void update() {
                this.pendingThreads = HikariPool.this.getThreadsAwaitingConnection();
                this.idleConnections = HikariPool.this.getIdleConnections();
                this.totalConnections = HikariPool.this.getTotalConnections();
                this.activeConnections = HikariPool.this.getActiveConnections();
            }
        };
    }

    public static class PoolInitializationException
    extends RuntimeException {
        private static final long serialVersionUID = 929872118275916520L;

        public PoolInitializationException(Throwable t) {
            super("Exception during pool initialization: " + t.getMessage(), t);
        }
    }

    private class HouseKeeper
    implements Runnable {
        private volatile long previous = HikariPool.access$700().currentTime();

        private HouseKeeper() {
        }

        @Override
        public void run() {
            List idleList;
            int removable;
            HikariPool.this.connectionTimeout = HikariPool.this.config.getConnectionTimeout();
            HikariPool.this.leakTask.updateLeakDetectionThreshold(HikariPool.this.config.getLeakDetectionThreshold());
            long now = clockSource.currentTime();
            long idleTimeout = HikariPool.this.config.getIdleTimeout();
            if (now < this.previous || now > clockSource.plusMillis(this.previous, 2L * HikariPool.this.HOUSEKEEPING_PERIOD_MS)) {
                LOGGER.warn("{} - Unusual system clock change detected, soft-evicting connections from pool.", (Object)HikariPool.this.poolName);
                this.previous = now;
                HikariPool.this.softEvictConnections();
                HikariPool.this.fillPool();
                return;
            }
            this.previous = now;
            HikariPool.this.logPoolState("Before cleanup\t");
            if (idleTimeout > 0L && (removable = (idleList = HikariPool.this.connectionBag.values(0)).size() - HikariPool.this.config.getMinimumIdle()) > 0) {
                Collections.sort(idleList, PoolEntry.LASTACCESS_COMPARABLE);
                Iterator iter = idleList.iterator();
                do {
                    PoolEntry poolEntry = (PoolEntry)iter.next();
                    if (clockSource.elapsedMillis(poolEntry.lastAccessed, now) <= idleTimeout || !HikariPool.this.connectionBag.reserve(poolEntry)) continue;
                    HikariPool.this.closeConnection(poolEntry, "(connection passed idleTimeout)");
                    --removable;
                } while (removable > 0 && iter.hasNext());
            }
            HikariPool.this.logPoolState("After cleanup\t");
            HikariPool.this.fillPool();
        }
    }

    private class PoolEntryCreator
    implements Callable<Boolean> {
        private PoolEntryCreator() {
        }

        @Override
        public Boolean call() throws Exception {
            long sleepBackoff = 200L;
            while (HikariPool.this.poolState == 0 && HikariPool.this.totalConnections.get() < HikariPool.this.config.getMaximumPoolSize()) {
                PoolEntry poolEntry = HikariPool.this.createPoolEntry();
                if (poolEntry != null) {
                    HikariPool.this.totalConnections.incrementAndGet();
                    HikariPool.this.connectionBag.add(poolEntry);
                    return Boolean.TRUE;
                }
                UtilityElf.quietlySleep(sleepBackoff);
                sleepBackoff = Math.min(HikariPool.this.connectionTimeout / 2L, (long)((double)sleepBackoff * 1.3));
            }
            return Boolean.FALSE;
        }
    }
}

